/*
 * Decompiled with CFR 0.152.
 */
package org.das2.math.filter;

import org.das2.datum.Datum;
import org.das2.datum.Units;
import org.das2.math.filter.AbstractFilter;
import org.das2.qds.QDataSet;

public class Butterworth
extends AbstractFilter {
    private int n;
    public static final String SYNOPSIS = "sikoried, 7/5/2011\nApply a Butterworth lowpass/bandpass filter of given order.\nusage: sampled.filters.Butterworth format order cutoff1 [cutoff2] < input > output\n  format : ssg/8 or ssg/16\n  order  : typically 3\n  cutoff1: cutoff frequency (lowpass)\n  cutoff2: cutoff frequency (bandpass)";

    public Butterworth(QDataSet source, int order, Datum freq, boolean lowp) {
        super(source);
        this.n = order;
        double ff = 2.0 * freq.doubleValue(Units.hertz) / this.getSampleRate(source, Units.hertz);
        double scale = Butterworth.computeScale(this.n, ff, lowp);
        double[] b = Butterworth.computeB(this.n, lowp);
        int i = 0;
        while (i < b.length) {
            int n = i++;
            b[n] = b[n] * scale;
        }
        double[] a = Butterworth.computeA(this.n, ff);
        this.setCoefficients(b, a);
    }

    public Butterworth(QDataSet source, int order, Datum freq1, Datum freq2, boolean pass) {
        super(source);
        this.n = order;
        double ff1 = 2.0 * freq1.doubleValue(Units.hertz) / this.getSampleRate(source, Units.hertz);
        double ff2 = 2.0 * freq2.doubleValue(Units.hertz) / this.getSampleRate(source, Units.hertz);
        double scale = Butterworth.computeScale(this.n, ff1, ff2, pass);
        double[] b = Butterworth.computeB(this.n, ff1, ff2, pass);
        int i = 0;
        while (i < b.length) {
            int n = i++;
            b[n] = b[n] * scale;
        }
        double[] a = Butterworth.computeA(this.n, ff1, ff2, pass);
        this.setCoefficients(b, a);
    }

    private static double[] computeB(int n, boolean lowp) {
        int i;
        double[] ccof = new double[n + 1];
        ccof[0] = 1.0;
        ccof[1] = n;
        for (i = 2; i < n / 2 + 1; ++i) {
            ccof[i] = (double)(n - i + 1) * ccof[i - 1] / (double)i;
            ccof[n - i] = ccof[i];
        }
        ccof[n - 1] = n;
        ccof[n] = 1.0;
        if (!lowp) {
            for (i = 1; i < n + 1; i += 2) {
                ccof[i] = -ccof[i];
            }
        }
        return ccof;
    }

    private static double[] computeB(int n, double f1, double f2, boolean pass) {
        double[] ccof = new double[2 * n + 1];
        if (pass) {
            double[] tcof = Butterworth.computeB(n, false);
            for (int i = 0; i < n; ++i) {
                ccof[2 * i] = tcof[i];
                ccof[2 * i + 1] = 0.0;
            }
            ccof[2 * n] = tcof[n];
        } else {
            double alpha = -2.0 * Math.cos(Math.PI * (f2 + f1) / 2.0) / Math.cos(Math.PI * (f2 - f1) / 2.0);
            ccof[0] = 1.0;
            ccof[1] = alpha;
            ccof[2] = 1.0;
            for (int i = 1; i < n; ++i) {
                int n2 = 2 * i + 2;
                ccof[n2] = ccof[n2] + ccof[2 * i];
                for (int j = 2 * i; j > 1; --j) {
                    int n3 = j + 1;
                    ccof[n3] = ccof[n3] + (alpha * ccof[j] + ccof[j - 1]);
                }
                ccof[2] = ccof[2] + (alpha * ccof[1] + 1.0);
                ccof[1] = ccof[1] + alpha;
            }
        }
        return ccof;
    }

    private static double[] computeA(int n, double f) {
        double[] rcof = new double[2 * n];
        double theta = Math.PI * f;
        double st = Math.sin(theta);
        double ct = Math.cos(theta);
        for (int k = 0; k < n; ++k) {
            double parg = Math.PI * (double)(2 * k + 1) / (double)(2 * n);
            double sparg = Math.sin(parg);
            double cparg = Math.cos(parg);
            double a = 1.0 + st * sparg;
            rcof[2 * k] = -ct / a;
            rcof[2 * k + 1] = -st * cparg / a;
        }
        double[] temp = Butterworth.binomialMult(rcof);
        double[] dcof = new double[n + 1];
        dcof[0] = 1.0;
        dcof[1] = temp[0];
        dcof[2] = temp[2];
        for (int k = 3; k < n + 1; ++k) {
            dcof[k] = temp[2 * k - 2];
        }
        return dcof;
    }

    private static double[] computeA(int n, double f1, double f2, boolean pass) {
        double cp = Math.cos(Math.PI * (f2 + f1) / 2.0);
        double theta = Math.PI * (f2 - f1) / 2.0;
        double st = Math.sin(theta);
        double ct = Math.cos(theta);
        double s2t = 2.0 * st * ct;
        double c2t = 2.0 * ct * ct - 1.0;
        double[] rcof = new double[2 * n];
        double[] tcof = new double[2 * n];
        for (int k = 0; k < n; ++k) {
            double parg = Math.PI * (double)(2 * k + 1) / (double)(2 * n);
            double sparg = Math.sin(parg);
            double cparg = Math.cos(parg);
            double a = 1.0 + s2t * sparg;
            rcof[2 * k] = c2t / a;
            rcof[2 * k + 1] = (pass ? 1.0 : -1.0) * s2t * cparg / a;
            tcof[2 * k] = -2.0 * cp * (ct + st * sparg) / a;
            tcof[2 * k + 1] = (pass ? -2.0 : 2.0) * cp * st * cparg / a;
        }
        double[] temp = Butterworth.trinomialMult(tcof, rcof);
        double[] dcof = new double[2 * n + 1];
        dcof[0] = 1.0;
        dcof[1] = temp[0];
        dcof[2] = temp[2];
        for (int k = 3; k < 2 * n + 1; ++k) {
            dcof[k] = temp[2 * k - 2];
        }
        return dcof;
    }

    private static double computeScale(int n, double f, boolean lowp) {
        double omega = Math.PI * f;
        double fomega = Math.sin(omega);
        double parg0 = Math.PI / (double)(2 * n);
        double sf = 1.0;
        for (int k = 0; k < n / 2; ++k) {
            sf *= 1.0 + fomega * Math.sin((double)(2 * k + 1) * parg0);
        }
        double d = fomega = lowp ? Math.sin(omega / 2.0) : Math.cos(omega / 2.0);
        if (n % 2 != 0) {
            sf *= fomega + (lowp ? Math.cos(omega / 2.0) : Math.sin(omega / 2.0));
        }
        sf = Math.pow(fomega, n) / sf;
        return sf;
    }

    private static double computeScale(int n, double f1, double f2, boolean pass) {
        double tt = Math.tan(Math.PI * (f2 - f1) / 2.0);
        if (pass) {
            tt = 1.0 / tt;
        }
        double sfr = 1.0;
        double sfi = 0.0;
        for (int k = 0; k < n; ++k) {
            double parg = Math.PI * (double)(2 * k + 1) / (double)(2 * n);
            double sparg = tt + Math.sin(parg);
            double cparg = Math.cos(parg);
            double a = (sfr + sfi) * (sparg - cparg);
            double b = sfr * sparg;
            double c = -sfi * cparg;
            sfr = b - c;
            sfi = a - b - c;
        }
        return 1.0 / sfr;
    }

    private static double[] binomialMult(double[] p) {
        int n = p.length / 2;
        double[] a = new double[2 * n];
        for (int i = 0; i < n; ++i) {
            for (int j = i; j > 0; --j) {
                int n2 = 2 * j;
                a[n2] = a[n2] + (p[2 * i] * a[2 * (j - 1)] - p[2 * i + 1] * a[2 * (j - 1) + 1]);
                int n3 = 2 * j + 1;
                a[n3] = a[n3] + (p[2 * i] * a[2 * (j - 1) + 1] + p[2 * i + 1] * a[2 * (j - 1)]);
            }
            a[0] = a[0] + p[2 * i];
            a[1] = a[1] + p[2 * i + 1];
        }
        return a;
    }

    private static double[] trinomialMult(double[] b, double[] c) {
        int n = b.length / 2;
        double[] a = new double[4 * n];
        a[0] = b[0];
        a[1] = b[1];
        a[2] = c[0];
        a[3] = c[1];
        for (int i = 1; i < n; ++i) {
            int n2 = 2 * (2 * i + 1);
            a[n2] = a[n2] + (c[2 * i] * a[2 * (2 * i - 1)] - c[2 * i + 1] * a[2 * (2 * i - 1) + 1]);
            int n3 = 2 * (2 * i + 1) + 1;
            a[n3] = a[n3] + (c[2 * i] * a[2 * (2 * i - 1) + 1] + c[2 * i + 1] * a[2 * (2 * i - 1)]);
            for (int j = 2 * i; j > 1; --j) {
                int n4 = 2 * j;
                a[n4] = a[n4] + (b[2 * i] * a[2 * (j - 1)] - b[2 * i + 1] * a[2 * (j - 1) + 1] + c[2 * i] * a[2 * (j - 2)] - c[2 * i + 1] * a[2 * (j - 2) + 1]);
                int n5 = 2 * j + 1;
                a[n5] = a[n5] + (b[2 * i] * a[2 * (j - 1) + 1] + b[2 * i + 1] * a[2 * (j - 1)] + c[2 * i] * a[2 * (j - 2) + 1] + c[2 * i + 1] * a[2 * (j - 2)]);
            }
            a[2] = a[2] + (b[2 * i] * a[0] - b[2 * i + 1] * a[1] + c[2 * i]);
            a[3] = a[3] + (b[2 * i] * a[1] + b[2 * i + 1] * a[0] + c[2 * i + 1]);
            a[0] = a[0] + b[2 * i];
            a[1] = a[1] + b[2 * i + 1];
        }
        return a;
    }

    @Override
    public String toString() {
        return "Butterworth filter n = " + this.n + " " + super.toString();
    }
}

