/*
 * Decompiled with CFR 0.152.
 */
package org.das2.datum;

import java.text.ParseException;
import java.util.Arrays;
import java.util.Locale;
import java.util.StringTokenizer;
import java.util.logging.Logger;
import org.das2.datum.Datum;
import org.das2.datum.DatumRange;
import org.das2.datum.DatumRangeUtil;
import org.das2.datum.LoggerManager;
import org.das2.datum.TimeLocationUnits;
import org.das2.datum.Units;
import org.das2.datum.UnitsConverter;
import org.das2.datum.format.TimeDatumFormatter;

public final class TimeUtil {
    private static final Logger logger = LoggerManager.getLogger("das2.datum");
    private static final int[][] daysInMonth = new int[][]{{0, 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31, 0}, {0, 31, 29, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31, 0}};
    private static final int[][] dayOffset = new int[][]{{0, 0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334, 365}, {0, 0, 31, 60, 91, 121, 152, 182, 213, 244, 274, 305, 335, 366}};
    public static final int YEAR = 1;
    public static final int MONTH = 2;
    public static final int DAY = 3;
    public static final int HOUR = 4;
    public static final int MINUTE = 5;
    public static final int SECOND = 6;
    public static final int MILLI = 7;
    public static final int MICRO = 8;
    public static final int NANO = 9;
    public static final int WEEK = 97;
    public static final int QUARTER = 98;
    public static final int HALF_YEAR = 99;
    public static final TimeDigit TD_YEAR = new TimeDigit(1, "YEAR", 12);
    public static final TimeDigit TD_MONTH = new TimeDigit(2, "MONTH", 30);
    public static final TimeDigit TD_DAY = new TimeDigit(3, "DAY", 24);
    public static final TimeDigit TD_HOUR = new TimeDigit(4, "HOUR", 60);
    public static final TimeDigit TD_MINUTE = new TimeDigit(5, "MINUTE", 60);
    public static final TimeDigit TD_SECOND = new TimeDigit(6, "SECOND", 1000);
    public static final TimeDigit TD_MILLI = new TimeDigit(7, "MILLISECONDS", 1000);
    public static final TimeDigit TD_MICRO = new TimeDigit(8, "MICROSECONDS", 1000);
    public static final TimeDigit TD_NANO = new TimeDigit(9, "NANOSECONDS", 1000);
    private static final String[] mons = new String[]{"Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"};

    private TimeUtil() {
    }

    public static int daysInMonth(int month, int year) {
        return daysInMonth[TimeUtil.isLeapYear(year) ? 1 : 0][month];
    }

    public static int julianDayIMCCE(int YY, int MM, int DD) {
        int GGG = 1;
        if (YY < 1582) {
            GGG = 0;
        }
        if (YY <= 1582 && MM < 10) {
            GGG = 0;
        }
        if (YY <= 1582 && MM == 10 && DD < 5) {
            GGG = 0;
        }
        int JD = -1 * (7 * ((MM + 9) / 12 + YY) / 4);
        int S = 1;
        if (MM - 9 < 0) {
            S = -1;
        }
        int A = Math.abs(MM - 9);
        int J1 = YY + S * (A / 7);
        J1 = -1 * ((J1 / 100 + 1) * 3 / 4);
        JD = JD + 275 * MM / 9 + DD + GGG * J1;
        JD = JD + 1721027 + 2 * GGG + 367 * YY;
        return JD;
    }

    public static int julianDay(int year, int month, int day) {
        if (year <= 1582) {
            throw new IllegalArgumentException("year must be more than 1582");
        }
        int jd = 367 * year - 7 * (year + (month + 9) / 12) / 4 - 3 * ((year + (month - 9) / 7) / 100 + 1) / 4 + 275 * month / 9 + day + 1721029;
        return jd;
    }

    public static int dayOfYear(int month, int day, int year) {
        return day + dayOffset[TimeUtil.isLeapYear(year) ? 1 : 0][month];
    }

    public static double getSecondsSinceMidnight(Datum datum) {
        double xx = datum.doubleValue(Units.t2000);
        if (xx < 0.0) {
            if ((xx %= 86400.0) == 0.0) {
                return 0.0;
            }
            return 86400.0 + xx;
        }
        return xx % 86400.0;
    }

    public static double getMicroSecondsSinceMidnight(Datum datum) {
        double xx = datum.doubleValue(Units.us2000);
        if (xx < 0.0) {
            if ((xx %= 8.64E10) == 0.0) {
                return 0.0;
            }
            return 8.64E10 + xx;
        }
        return xx % 8.64E10;
    }

    public static int getJulianDay(Datum datum) {
        double xx = datum.doubleValue(Units.mj1958);
        return (int)Math.floor(xx) + 2436205;
    }

    public static int getJulianDay(long val, Units units) {
        if (units == Units.us2000) {
            return (int)(val / 86400000000L) + 15340 + 2436205;
        }
        if (units == Units.t2000) {
            return (int)(val / 86400L) + 15340 + 2436205;
        }
        if (units == Units.mj1958) {
            return (int)val + 2436205;
        }
        UnitsConverter uc = units.getConverter(Units.mj1958);
        return (int)Math.floor(uc.convert(val)) + 2436205;
    }

    public static Datum toDatum(TimeStruct d) {
        int year = d.year;
        int month = d.month;
        int day = d.day;
        int jd = 367 * year - 7 * (year + (month + 9) / 12) / 4 - 3 * ((year + (month - 9) / 7) / 100 + 1) / 4 + 275 * month / 9 + day + 1721029;
        double us2000 = (double)(jd - 2451545) * 8.64E10;
        return Datum.create((double)d.hour * 3.6E9 + (double)d.minute * 6.0E7 + d.seconds * 1000000.0 + (double)(d.millis * 1000) + (double)d.micros + us2000, (Units)Units.us2000);
    }

    public static Datum toDatum(TimeStruct d, Units u) {
        int year = d.year;
        int month = d.month;
        int day = d.day;
        int jd = 367 * year - 7 * (year + (month + 9) / 12) / 4 - 3 * ((year + (month - 9) / 7) / 100 + 1) / 4 + 275 * month / 9 + day + 1721029;
        if (u == Units.cdfTT2000) {
            double us2000 = (double)(jd - 2451545) * 8.64E10;
            double tt2000 = Units.us2000.convertDoubleTo(Units.cdfTT2000, us2000);
            Units.cdfTT2000.createDatum(tt2000);
            Datum rtt2000 = Datum.create((double)d.hour * 3.6E12 + (double)d.minute * 6.0E10 + d.seconds * 1.0E9 + (double)d.millis * 1000000.0 + (double)d.micros * 1000.0 + tt2000, (Units)Units.cdfTT2000);
            return rtt2000;
        }
        if (u != Units.us2000) {
            double us2000 = (double)(jd - 2451545) * 8.64E10;
            Datum rus2000 = Datum.create((double)d.hour * 3.6E9 + (double)d.minute * 6.0E7 + d.seconds * 1000000.0 + (double)(d.millis * 1000) + (double)d.micros + us2000, (Units)Units.us2000);
            return rus2000.convertTo(u);
        }
        double us2000 = (double)(jd - 2451545) * 8.64E10;
        return Datum.create((double)d.hour * 3.6E9 + (double)d.minute * 6.0E7 + d.seconds * 1000000.0 + (double)(d.millis * 1000) + (double)d.micros + us2000, (Units)Units.us2000);
    }

    public static TimeStruct julianToGregorian(int julian) {
        int j = julian + 32044;
        int g = j / 146097;
        int dg = j % 146097;
        int c = (dg / 36524 + 1) * 3 / 4;
        int dc = dg - c * 36524;
        int b = dc / 1461;
        int db = dc % 1461;
        int a = (db / 365 + 1) * 3 / 4;
        int da = db - a * 365;
        int y = g * 400 + c * 100 + b * 4 + a;
        int m = (da * 5 + 308) / 153 - 2;
        int d = da - (m + 4) * 153 / 5 + 122;
        int Y = y - 4800 + (m + 2) / 12;
        int M = (m + 2) % 12 + 1;
        int D = d + 1;
        TimeStruct result = new TimeStruct();
        result.year = Y;
        result.month = M;
        result.day = D;
        result.isLocation = true;
        return result;
    }

    public static TimeStruct toTimeStruct(Datum datum) {
        int minute;
        Units u = datum.getUnits();
        double d = datum.doubleValue(u);
        int mjd1958 = (int)datum.doubleValue(Units.mj1958);
        if (mjd1958 < -714781) {
            throw new IllegalArgumentException("invalid time: mjd1958=" + mjd1958);
        }
        if (mjd1958 > 2937613) {
            throw new IllegalArgumentException("invalid time: mjd1958=" + mjd1958);
        }
        double midnight = Units.mj1958.convertDoubleTo(u, mjd1958);
        double sinceMidnight = d - midnight;
        int jd = 2436205 + mjd1958;
        if (u == Units.cdfTT2000 && sinceMidnight < 0.0) {
            boolean isLeap;
            TimeStruct result = TimeUtil.julianToGregorian(jd);
            boolean bl = isLeap = result.month == 1 && result.day == 1 || result.month == 1 && result.day == 1;
            if (isLeap) {
                jd = 2436205 + --mjd1958;
                sinceMidnight += 8.6401E13;
            }
        }
        double nanoseconds = u.getOffsetUnits().convertDoubleTo(Units.nanoseconds, sinceMidnight);
        if (jd < 0) {
            throw new IllegalArgumentException("julian day is negative.");
        }
        if (nanoseconds < 0.0) {
            --jd;
            nanoseconds += 8.64E13;
        }
        if (nanoseconds >= 8.64E13 && u != Units.cdfTT2000) {
            ++jd;
            nanoseconds -= 8.64E13;
        }
        TimeStruct result = TimeUtil.julianToGregorian(jd);
        int hour = (int)(nanoseconds / 3.6E12);
        if (hour > 23) {
            hour = 23;
        }
        if ((minute = (int)((nanoseconds - (double)hour * 3.6E12) / 6.0E10)) > 59) {
            minute = 59;
        }
        double justNanoSeconds = nanoseconds - (double)hour * 3.6E12 - (double)minute * 6.0E10;
        result.doy = TimeUtil.dayOfYear(result.month, result.day, result.year);
        result.hour = hour;
        result.minute = minute;
        result.seconds = justNanoSeconds / 1.0E9;
        result.isLocation = true;
        return result;
    }

    public static TimeStruct add(TimeStruct a, TimeStruct b) {
        if (b.year > 1000 && a.year > 1000) {
            throw new IllegalArgumentException("cannot add more than 1000 years at a time.  Did you attempt to add two time locations?");
        }
        TimeStruct result = new TimeStruct();
        result.year = a.year + b.year;
        result.month = a.month + b.month;
        result.day = a.day + b.day;
        result.doy = a.doy + b.doy;
        result.hour = a.hour + b.hour;
        result.minute = a.minute + b.minute;
        result.seconds = a.seconds + b.seconds;
        result.millis = a.millis + b.millis;
        result.micros = a.micros + b.micros;
        result.isLocation = a.isLocation || b.isLocation;
        return result;
    }

    public static TimeStruct subtract(TimeStruct a, TimeStruct b) {
        TimeStruct result = new TimeStruct();
        result.year = a.year - b.year;
        result.month = a.month - b.month;
        result.day = a.day - b.day;
        result.doy = a.doy - b.doy;
        result.hour = a.hour - b.hour;
        result.minute = a.minute - b.minute;
        result.seconds = a.seconds - b.seconds;
        result.millis = a.millis - b.millis;
        result.micros = a.micros - b.micros;
        return result;
    }

    public static int[] toTimeArray(Datum time) {
        TimeStruct ts = TimeUtil.toTimeStruct(time);
        int seconds = (int)(ts.seconds + 5.0E-7);
        int micros = (int)((ts.seconds + 5.0E-7 - (double)seconds) * 1000000.0);
        int millis = micros / 1000;
        micros = micros - millis * 1000 + ts.micros + ts.millis * 1000;
        return new int[]{ts.year, ts.month, ts.day, ts.hour, ts.minute, seconds, millis, micros};
    }

    public static int[] fromDatum(Datum time) {
        TimeStruct ts = TimeUtil.toTimeStruct(time);
        int seconds = (int)(ts.seconds + 5.0E-10);
        int nanos = (int)((ts.seconds + 5.0E-10 - (double)seconds) * 1.0E8);
        return new int[]{ts.year, ts.month, ts.day, ts.hour, ts.minute, seconds, nanos};
    }

    public static Datum toDatum(int[] timeArray) {
        int year = timeArray[0];
        int month = timeArray[1];
        int day = timeArray[2];
        if (month < 1) {
            logger.info("month was less than 0");
            month += 12;
            --year;
        }
        if (month > 12) {
            month -= 12;
            ++year;
        }
        if (month < 1) {
            throw new IllegalArgumentException("month is less than 1");
        }
        if (month > 12) {
            throw new IllegalArgumentException("month is greater than 12");
        }
        int jd = 367 * year - 7 * (year + (month + 9) / 12) / 4 - 3 * ((year + (month - 9) / 7) / 100 + 1) / 4 + 275 * month / 9 + day + 1721029;
        int hour = timeArray[3];
        int minute = timeArray[4];
        double seconds = (float)timeArray[5] + (float)hour * 3600.0f + (float)minute * 60.0f;
        if (timeArray.length > 6) {
            seconds += (double)timeArray[6] / 1.0E9;
        }
        double us2000 = UnitsConverter.getConverter(Units.mj1958, Units.us2000).convert((double)(jd - 2436205) + seconds / 86400.0);
        return Datum.create(us2000, (Units)Units.us2000);
    }

    public static Datum toDatumDuration(int[] timeArray) {
        int year = timeArray[0];
        int month = timeArray[1];
        int day = timeArray[2];
        int days = day + month * 30 + year * 365;
        if (timeArray.length == 7) {
            if (days == 0) {
                return Units.seconds.createDatum((double)(timeArray[3] * 3600 + timeArray[4] * 60 + timeArray[5]) + (double)timeArray[6] / 1.0E9);
            }
            return Units.days.createDatum((double)days + (double)timeArray[3] / 24.0 + (double)timeArray[4] / 1440.0 + (double)timeArray[5] / 86400.0 + (double)timeArray[6] / 8.64E13);
        }
        if (days == 0) {
            return Units.seconds.createDatum(timeArray[3] * 3600 + timeArray[4] * 60 + timeArray[5]);
        }
        return Units.days.createDatum((double)days + (double)timeArray[3] / 24.0 + (double)timeArray[4] / 1440.0 + (double)timeArray[5] / 86400.0);
    }

    public static Datum toDatum(int[] timeArray, Units u) {
        int nanos;
        int year = timeArray[0];
        int month = timeArray[1];
        int day = timeArray[2];
        int jd = 367 * year - 7 * (year + (month + 9) / 12) / 4 - 3 * ((year + (month - 9) / 7) / 100 + 1) / 4 + 275 * month / 9 + day + 1721029;
        int n = nanos = timeArray.length == 6 ? 0 : timeArray[6];
        if (u == Units.cdfTT2000) {
            double us2000 = (double)(jd - 2451545) * 8.64E10;
            double tt2000 = Units.us2000.convertDoubleTo(Units.cdfTT2000, us2000);
            Units.cdfTT2000.createDatum(tt2000);
            Datum rtt2000 = Datum.create((double)timeArray[3] * 3.6E12 + (double)timeArray[4] * 6.0E10 + (double)timeArray[5] * 1.0E9 + (double)nanos + tt2000, (Units)Units.cdfTT2000);
            return rtt2000;
        }
        if (u != Units.us2000) {
            double us2000 = (double)(jd - 2451545) * 8.64E10;
            Datum rus2000 = Datum.create((double)timeArray[3] * 3.6E9 + (double)timeArray[4] * 6.0E7 + (double)timeArray[5] * 1000000.0 + (double)nanos / 1000.0 + us2000, (Units)Units.us2000);
            return rus2000.convertTo(u);
        }
        double us2000 = (double)(jd - 2451545) * 8.64E10;
        Datum rus2000 = Datum.create((double)timeArray[3] * 3.6E9 + (double)timeArray[4] * 6.0E7 + (double)timeArray[5] * 1000000.0 + (double)nanos / 1000.0 + us2000, (Units)Units.us2000);
        return rus2000;
    }

    public static boolean isLeapYear(int year) {
        return year % 4 == 0 && (year % 400 == 0 || year % 100 != 0);
    }

    public static TimeStruct carry(TimeStruct t) {
        TimeStruct result = t;
        boolean isLeap = false;
        if (result.seconds >= 60.0) {
            if (result.month == 6 && result.day == 30 || result.month == 12 && result.day == 31) {
                isLeap = true;
            }
            if (result.hour < 23 || result.minute < 59 || !isLeap) {
                result.seconds -= 60.0;
                ++result.minute;
            }
        }
        if (result.minute >= 60) {
            result.minute -= 60;
            ++result.hour;
        }
        if (result.hour >= 24) {
            result.hour -= 24;
            ++result.day;
        }
        while (result.month > 12) {
            result.month -= 12;
            ++result.year;
        }
        int daysThisMonth = TimeUtil.daysInMonth(result.month, result.year);
        if (result.day > daysThisMonth) {
            result.day -= daysThisMonth;
            ++result.month;
        }
        if (result.month > 12) {
            result.month -= 12;
            ++result.year;
        }
        return result;
    }

    public static TimeStruct borrow(TimeStruct t) {
        TimeStruct result = t;
        if (result.seconds < 0.0) {
            result.seconds += 60.0;
            --result.minute;
        }
        if (result.minute < 0) {
            result.minute += 60;
            --result.hour;
        }
        if (result.hour < 0) {
            result.hour += 24;
            --result.day;
        }
        if (result.day < 0 || result.month < 1) {
            throw new IllegalArgumentException("Borrow operation not defined for months<1 or days<0");
        }
        if (result.day == 0) {
            int daysLastMonth = result.month > 1 ? TimeUtil.daysInMonth(result.month - 1, result.year) : 31;
            result.day += daysLastMonth;
            --result.month;
        }
        if (result.month == 0) {
            result.month += 12;
            --result.year;
        }
        return result;
    }

    public static TimeStruct normalize(TimeStruct t) {
        if (t.doy > 0 && t.day == 0) {
            int leap;
            int n = leap = TimeUtil.isLeapYear(t.year) ? 1 : 0;
            if (t.doy > dayOffset[leap][13]) {
                throw new IllegalArgumentException("doy>" + dayOffset[leap][13] + ")");
            }
            int month = 12;
            while (dayOffset[leap][month] > t.doy) {
                --month;
            }
            t.day = t.doy - dayOffset[leap][month];
            t.month = month;
        }
        return TimeUtil.carry(TimeUtil.borrow(t));
    }

    public static TimeStruct roundNDigits(TimeStruct ts, int n) {
        int roundMicros;
        if (n > 6) {
            throw new IllegalArgumentException("only 0 to 6 digits supported");
        }
        double fracSeconds = ts.seconds - (double)((int)ts.seconds);
        ts.seconds = (int)ts.seconds;
        ts.micros += ts.millis * 1000;
        ts.millis = 0;
        double pow = Math.pow(10.0, 6 - n);
        ts.micros = roundMicros = (int)((double)Math.round(((double)ts.micros + 1000000.0 * fracSeconds) / pow) * pow);
        if (ts.micros >= 1000000) {
            ts.micros -= 1000000;
            ts.seconds += 1.0;
        }
        return TimeUtil.normalize(ts);
    }

    public static Datum next(TimeDigit td, int count, Datum datum) {
        if (td == TD_NANO) {
            throw new IllegalArgumentException("not supported nanos");
        }
        TimeStruct array = TimeUtil.toTimeStruct(datum);
        int step = td.getOrdinal();
        switch (td.getOrdinal()) {
            case 7: {
                array.millis += count;
                break;
            }
            case 6: {
                array.seconds += (double)count;
                break;
            }
            case 5: {
                array.minute += count;
                break;
            }
            case 4: {
                array.hour += count;
                break;
            }
            case 3: {
                array.day += count;
                break;
            }
            case 2: {
                array.month += count;
                array.day = 1;
                break;
            }
            case 1: {
                array.year += count;
                array.month = 1;
                array.day = 1;
                break;
            }
        }
        if (step < 7) {
            array.millis = 0;
        }
        if (step < 6) {
            array.seconds = 0.0;
        }
        if (step < 5) {
            array.minute = 0;
        }
        if (step < 4) {
            array.hour = 0;
        }
        if (array.month > 12) {
            ++array.year;
            array.month -= 12;
        }
        Datum result = TimeUtil.toDatum(array);
        return result;
    }

    private static TimeStruct next(int step, TimeStruct array) {
        switch (step) {
            case 6: {
                array.seconds += 1.0;
                break;
            }
            case 5: {
                ++array.minute;
                break;
            }
            case 4: {
                ++array.hour;
                break;
            }
            case 3: {
                ++array.day;
                break;
            }
            case 2: {
                ++array.month;
                array.day = 1;
                break;
            }
            case 98: {
                array.month = (array.month - 1 + 3) / 3 * 3 + 1;
                array.day = 1;
                break;
            }
            case 99: {
                array.month = (array.month - 1 + 6) / 6 * 6 + 1;
                array.day = 1;
                break;
            }
            case 1: {
                ++array.year;
                array.month = 1;
                array.day = 1;
                break;
            }
        }
        if (step < 4) {
            array.hour = 0;
            array.minute = 0;
            array.seconds = 0.0;
        }
        if (array.month > 12) {
            ++array.year;
            array.month -= 12;
        }
        return array;
    }

    public static Datum next(int step, Datum datum) {
        if (step == 9) {
            throw new IllegalArgumentException("not supported nanos");
        }
        return TimeUtil.toDatum(TimeUtil.next(step, TimeUtil.toTimeStruct(datum)));
    }

    public static Datum ceil(int step, Datum datum) {
        Datum next = TimeUtil.next(step, datum);
        Datum t1 = TimeUtil.prev(step, next);
        if (t1.equals(datum)) {
            return datum;
        }
        return next;
    }

    public static Datum floor(int step, Datum datum) {
        Datum prev = TimeUtil.prev(step, datum);
        Datum t1 = TimeUtil.next(step, prev);
        if (t1.equals(datum)) {
            return datum;
        }
        return prev;
    }

    public static Datum nextMonth(Datum datum) {
        return TimeUtil.next(2, datum);
    }

    public static Datum prevWeek(Datum datum) {
        TimeStruct t = TimeUtil.toTimeStruct(datum);
        t.day -= 7;
        if (t.day < 1) {
            --t.month;
            t.day += TimeUtil.daysInMonth(t.month, t.year);
        }
        return TimeUtil.toDatum(t);
    }

    public static DatumRange dayContaining(Datum t) {
        Datum midnight = TimeUtil.prevMidnight(t);
        return new DatumRange(midnight, TimeUtil.next(3, midnight));
    }

    public static Datum prev(int step, Datum datum) {
        TimeStruct t = TimeUtil.toTimeStruct(datum);
        switch (step) {
            case 97: {
                throw new IllegalArgumentException("not supported, use prevWeek");
            }
            case 1: {
                t.month = 1;
            }
            case 99: {
                t.month = (t.month - 1) / 6 * 6 + 1;
            }
            case 98: {
                t.month = (t.month - 1) / 3 * 3 + 1;
            }
            case 2: {
                t.day = 1;
            }
            case 3: {
                t.hour = 0;
            }
            case 4: {
                t.minute = 0;
            }
            case 5: {
                t.seconds = 0.0;
            }
            case 6: {
                t.seconds = (int)t.seconds;
            }
        }
        Datum result = TimeUtil.toDatum(t);
        if (result.equals(datum)) {
            Datum d = datum.subtract(500.0, Units.milliseconds);
            if (d.equals(result)) {
                throw new IllegalStateException("aborting to avoid stack overflow!");
            }
            return TimeUtil.prev(step, d);
        }
        return result;
    }

    public static Datum now() {
        double us2000 = ((double)System.currentTimeMillis() - 9.466848E11) * 1000.0;
        return Units.us2000.createDatum(us2000);
    }

    public static double convert(int year, int month, int day, int hour, int minute, double second, TimeLocationUnits units) {
        int jd;
        if (month > 0) {
            jd = TimeUtil.julianDay(year, month, day);
        } else {
            int month1 = 1;
            int day1 = 1;
            jd = TimeUtil.julianDay(year, month1, day1);
            jd += day - 1;
        }
        double us2000 = (double)(jd - 2451545) * 8.64E10 + (second += (double)hour * 3600.0 + (double)minute * 60.0) * 1000000.0;
        if (units == Units.us2000) {
            return us2000;
        }
        return Units.us2000.convertDoubleTo(units, us2000);
    }

    public static int monthNumber(String s) throws ParseException {
        if (s.length() < 3) {
            throw new ParseException("need at least three letters", 0);
        }
        s = s.substring(0, 3);
        for (int i = 0; i < 12; ++i) {
            if (!s.equalsIgnoreCase(mons[i])) continue;
            return i + 1;
        }
        throw new ParseException("Unable to parse month", 0);
    }

    public static String monthNameAbbrev(int mon) {
        if (mon < 1 || mon > 12) {
            throw new IllegalArgumentException("invalid month number: " + mon);
        }
        return mons[mon - 1];
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public static TimeStruct parseTime(String s) throws ParseException {
        int leap;
        int i;
        int len;
        int n;
        int end_of_date;
        boolean DATE = false;
        boolean YEAR = true;
        int MONTH = 2;
        int DAY = 3;
        int HOUR = 4;
        int MINUTE = 5;
        int SECOND = 6;
        String DELIMITERS = " \t/-:,_;";
        String PDSDELIMITERS = " \t/-T:,_;";
        String[] months = new String[]{"january", "febuary", "march", "april", "may", "june", "july", "august", "september", "october", "november", "december"};
        String[] tok = new String[10];
        boolean[] want = new boolean[7];
        Arrays.fill(want, false);
        s = s.trim();
        if (s.length() == 0) {
            throw new ParseException("string is empty", 0);
        }
        if (s.charAt(0) == '-') {
            throw new ParseException("string starts with minus sign", 0);
        }
        if (s.equals("now")) {
            return TimeUtil.toTimeStruct(TimeUtil.now());
        }
        String delimiters = " \t/-:,_;";
        int c = s.indexOf(90);
        if (c != -1) {
            s = s.substring(0, c);
        }
        if ((end_of_date = s.indexOf(84)) > 1) {
            c = end_of_date - 1;
            if (Character.isDigit(s.charAt(c))) {
                delimiters = " \t/-T:,_;";
            } else {
                end_of_date = -1;
            }
        }
        if (end_of_date == -1) {
            n = 0;
            len = s.length();
            for (i = 0; i < len; ++i) {
                c = delimiters.substring(2).indexOf(s.charAt(i));
                if (c != -1) {
                    ++n;
                }
                if (n != 3) continue;
                end_of_date = i;
                break;
            }
        }
        int year = 0;
        int month = 0;
        int day_month = 0;
        int day_year = 0;
        int hour = 0;
        int minute = 0;
        double second = 0.0;
        StringTokenizer st = new StringTokenizer(s, delimiters);
        if (!st.hasMoreTokens()) {
            throw new ParseException("No tokens in '" + s + "'", 0);
        }
        for (n = 0; n < 10 && st.hasMoreTokens(); ++n) {
            tok[n] = st.nextToken();
        }
        want[3] = true;
        want[2] = true;
        want[1] = true;
        want[0] = true;
        int hold = 0;
        int tokIndex = -1;
        for (i = 0; i < n; ++i) {
            double value;
            tokIndex = s.indexOf(tok[i], tokIndex + 1);
            if (end_of_date != -1 && want[0] && tokIndex > end_of_date) {
                want[0] = false;
                want[6] = true;
                want[5] = true;
                want[4] = true;
            }
            len = tok[i].length();
            try {
                if (tok[i].length() > 0 && Character.isLetter(tok[i].charAt(0))) {
                    throw new NumberFormatException("must start with a number: " + tok[i]);
                }
                value = Double.parseDouble(tok[i]);
            }
            catch (NumberFormatException e) {
                if (len < 3 || !want[0]) {
                    throw new ParseException("Error at token '" + tok[i] + "' in '" + s + "'", 0);
                }
                for (int j = 0; j < 12; ++j) {
                    if (!tok[i].equalsIgnoreCase(months[j]) && !tok[i].equalsIgnoreCase(mons[j])) continue;
                    month = j + 1;
                    want[2] = false;
                    if (hold <= 0) break;
                    if (day_month > 0) {
                        throw new ParseException("Ambiguous dates in token '" + tok[i] + "' in '" + s + "'", 0);
                    }
                    day_month = hold;
                    hold = 0;
                    want[3] = false;
                    break;
                }
                if (!want[2]) continue;
                throw new ParseException("Error at token '" + tok[i] + "' in '" + s + "'", 0);
            }
            if (Math.IEEEremainder(value, 1.0) != 0.0) {
                if (!want[6]) throw new ParseException("Error at token '" + tok[i] + "' in '" + s + "'", 0);
                second = value;
                break;
            }
            int number = (int)value;
            if (number < 0) {
                throw new ParseException("Error at token '" + tok[i] + "' in '" + s + "'", 0);
            }
            if (want[0]) {
                if (number == 0) {
                    throw new ParseException("m,d, or y can't be 0 in '" + s + "'", 0);
                }
                if (number >= 10000000 && want[1]) {
                    year = number / 10000;
                    want[1] = false;
                    month = number / 100 % 100;
                    want[2] = false;
                    day_month = number % 100;
                    day_year = 0;
                    want[3] = false;
                } else if (number >= 1000000 && want[1]) {
                    year = number / 1000;
                    want[1] = false;
                    day_year = number % 1000;
                    month = 0;
                    want[2] = false;
                    want[3] = false;
                } else if (number > 31) {
                    if (want[1]) {
                        if (hold != 0 && year < 100 && year > 50) {
                            throw new ParseException("Held digit (" + hold + ") before two-digit year (" + year + "): " + s, 0);
                        }
                        year = number;
                        if (year < 1000) {
                            year += 1900;
                        }
                        want[1] = false;
                    } else {
                        if (!want[2]) throw new ParseException("Error at token '" + tok[i] + "' in '" + s + "'", 0);
                        want[2] = false;
                        month = 0;
                        day_year = number;
                        want[3] = false;
                    }
                } else if (number > 12) {
                    if (!want[3]) throw new ParseException("Error at token '" + tok[i] + "' in '" + s + "'", 0);
                    if (hold > 0) {
                        month = hold;
                        want[2] = false;
                    }
                    if (len == 3) {
                        if (month > 0) {
                            throw new ParseException("Error at token '" + tok[i] + "' in '" + s + "'", 0);
                        }
                        day_year = number;
                        day_month = 0;
                        want[2] = false;
                    } else {
                        day_month = number;
                    }
                    want[3] = false;
                } else if (!want[2]) {
                    if (month > 0) {
                        day_month = number;
                        day_year = 0;
                    } else {
                        day_year = number;
                        day_month = 0;
                    }
                    want[3] = false;
                } else if (!want[3]) {
                    if (day_year > 0) {
                        throw new ParseException("Error at token '" + tok[i] + "' in '" + s + "'", 0);
                    }
                    month = number;
                    want[2] = false;
                } else if (!want[1]) {
                    if (len == 3) {
                        if (month > 0) {
                            throw new ParseException("Error at token '" + tok[i] + "' in '" + s + "'", 0);
                        }
                        day_year = number;
                        day_month = 0;
                        want[3] = false;
                    } else {
                        if (day_year > 0) {
                            throw new ParseException("Error at token '" + tok[i] + "' in '" + s + "'", 0);
                        }
                        month = number;
                        if (hold > 0) {
                            day_month = hold;
                            want[3] = false;
                        }
                    }
                    want[2] = false;
                } else if (hold > 0) {
                    month = hold;
                    hold = 0;
                    want[2] = false;
                    day_month = number;
                    want[3] = false;
                } else {
                    hold = number;
                }
                if (want[1] || want[2] || want[3]) continue;
                want[0] = false;
                want[6] = true;
                want[5] = true;
                want[4] = true;
                continue;
            }
            if (want[4]) {
                if (len == 4) {
                    hold = number / 100;
                    if (hold > 24) {
                        throw new ParseException("Error at token '" + tok[i] + "' in '" + s + "'", 0);
                    }
                    hour = hold;
                    hold = number % 100;
                    if (hold > 60) {
                        throw new ParseException("Error at token '" + tok[i] + "' in '" + s + "'", 0);
                    }
                    minute = hold;
                    want[5] = false;
                } else {
                    if (number > 24) {
                        throw new ParseException("Error at token '" + tok[i] + "' in '" + s + "'", 0);
                    }
                    hour = number;
                }
                want[4] = false;
                continue;
            }
            if (want[5]) {
                if (number > 60) {
                    throw new ParseException("Error at token '" + tok[i] + "' in '" + s + "'", 0);
                }
                minute = number;
                want[5] = false;
                continue;
            }
            if (!want[6]) throw new ParseException("Error at token '" + tok[i] + "' in '" + s + "'", 0);
            if (number > 61) {
                throw new ParseException("Error at token '" + tok[i] + "' in '" + s + "'", 0);
            }
            second = number;
            want[6] = false;
        }
        if (want[1]) {
            throw new ParseException("This doesn't appear to contain a year: '" + s + "'", 0);
        }
        if (month > 12) {
            throw new ParseException("Month is greater than 12 in '" + s + "'", 0);
        }
        if (month > 0 && day_month <= 0) {
            day_month = 1;
        }
        int n2 = (year & 3) > 0 ? 0 : (year % 100 > 0 ? 1 : (leap = year % 400 > 0 ? 0 : 1));
        if (month > 0 && day_month > 0 && day_year == 0) {
            if (day_month > daysInMonth[leap][month]) {
                throw new ParseException("day of month too high in '" + s + "'", 0);
            }
            day_year = dayOffset[leap][month] + day_month;
        } else if (day_year > 0 && month == 0 && day_month == 0) {
            if (day_year > 365 + leap) {
                throw new ParseException("day of year too high in '" + s + "'", 0);
            }
            for (i = 2; i < 14 && day_year > dayOffset[leap][i]; ++i) {
            }
            month = --i;
            day_month = day_year - dayOffset[leap][i];
        } else {
            if (month == 0) {
                month = 1;
            }
            day_month = 1;
        }
        TimeStruct result = new TimeStruct();
        result.year = year;
        result.month = month;
        result.day = day_month;
        result.doy = day_year;
        result.hour = hour;
        result.minute = minute;
        result.seconds = second;
        result.isLocation = true;
        result.want = want;
        return result;
    }

    public static Datum create(String s) throws ParseException {
        TimeStruct ts = TimeUtil.parseTime(s);
        return TimeUtil.toDatum(ts);
    }

    public static Datum createValid(String validString) {
        try {
            return TimeUtil.create(validString);
        }
        catch (ParseException ex) {
            throw new RuntimeException(ex);
        }
    }

    public static boolean isValidTime(String string) {
        try {
            TimeUtil.create(string);
            return true;
        }
        catch (ParseException ex) {
            return false;
        }
    }

    public static void main(String[] args) throws Exception {
        System.out.println("" + TimeUtil.isLeapYear(1900));
        System.out.println("" + TimeUtil.isLeapYear(2000));
        System.out.println("" + TimeUtil.isLeapYear(1996));
        System.out.println("" + TimeUtil.isLeapYear(1999));
        System.out.println("" + TimeUtil.isLeapYear(2100));
        System.out.println("TimeUtil.parse=" + TimeUtil.parseTime("2010"));
        System.out.println(TimeUtil.now());
        System.out.println(Datum.create(TimeUtil.convert(2000, 1, 2, 0, 0, 0.0, Units.us2000), (Units)Units.us2000));
        Datum x = TimeUtil.create("2000-1-1 0:00:33.45");
        System.out.println(x);
        TimeStruct ts = TimeUtil.toTimeStruct(x);
        System.out.println(TimeUtil.toDatum(ts));
        TimeDatumFormatter tf = TimeDatumFormatter.DEFAULT;
        for (int i = 0; i < 44; ++i) {
            System.out.println(tf.format(x) + "\t" + (long)x.doubleValue(Units.us2000));
            x = TimeUtil.prev(6, x);
        }
        Units[] uu = new Units[]{Units.cdfEpoch, Units.us1980, Units.us2000, Units.mj1958};
        for (int i = 0; i < uu.length; ++i) {
            Units u = uu[i];
            for (int j = 0; j < 10000; ++j) {
                Datum d = u.createDatum(j);
                String s = d.toString();
                ts = TimeUtil.toTimeStruct(d);
                Datum d1 = TimeUtil.toDatum(ts);
                if (d1.equals(d) || d1.subtract(d).doubleValue(Units.microseconds) < 1.0E-6) continue;
                System.err.println(d1.subtract(d));
                System.err.println("" + i + " " + j + ": " + d + " " + d1 + " " + ts);
            }
        }
    }

    public static Datum prevMidnight(Datum datum) {
        return datum.subtract(TimeUtil.getSecondsSinceMidnight(datum), Units.seconds);
    }

    public static Datum nextMidnight(Datum datum) {
        double d = TimeUtil.getMicroSecondsSinceMidnight(datum);
        if (d == 0.0) {
            return datum;
        }
        return TimeUtil.next(3, datum);
    }

    public static Datum createTimeDatum(int year, int month, int day, int hour, int minute, int second, int nano) {
        if (year < 1000) {
            throw new IllegalArgumentException("year must not be < 1000, and 2 digit years are not allowed(year=" + year + ")");
        }
        if (year > 9001) {
            throw new IllegalArgumentException("year must be smaller than 9000");
        }
        int jd = 367 * year - 7 * (year + (month + 9) / 12) / 4 - 3 * ((year + (month - 9) / 7) / 100 + 1) / 4 + 275 * month / 9 + day + 1721029;
        double microseconds = (double)second * 1000000.0 + (double)hour * 3.6E9 + (double)minute * 6.0E7 + (double)nano / 1000.0;
        double us2000 = UnitsConverter.getConverter(Units.mj1958, Units.us2000).convert(jd - 2436205) + microseconds;
        return Datum.create(us2000, (Units)Units.us2000);
    }

    public static class TimeDigit {
        int ordinal;
        String label;
        int divisions;
        private static final TimeDigit[] digits = new TimeDigit[10];

        public String toString() {
            return this.label;
        }

        private TimeDigit(int ordinal, String label, int divisions) {
            this.ordinal = ordinal;
            this.label = label;
            this.divisions = divisions;
            TimeDigit.digits[ordinal] = this;
        }

        public int getOrdinal() {
            return this.ordinal;
        }

        public int divisions() {
            return this.divisions;
        }

        public static TimeDigit fromOrdinal(int ordinal) {
            return digits[ordinal];
        }
    }

    public static final class TimeStruct {
        public int year;
        public int month;
        public int day;
        public int doy;
        public int hour;
        public int minute;
        public double seconds;
        public int millis;
        public int micros;
        public boolean isLocation = false;
        public boolean[] want;

        public String toString() {
            if (this.isLocation) {
                int dayOfYear = TimeUtil.dayOfYear(this.month, this.day, this.year);
                return String.format(Locale.US, "%4d/%02d/%02d %02d:%02d:%06.3f (doy=%03d)", this.year, this.month, this.day, this.hour, this.minute, this.seconds + (double)this.millis / 1000.0 + (double)this.micros / 1000000.0, dayOfYear);
            }
            int intSeconds = (int)this.seconds;
            int nanos = (int)(1.0E9 * (this.seconds - (double)intSeconds));
            nanos += this.micros * 1000;
            return DatumRangeUtil.formatISO8601Duration(new int[]{this.year, this.month, this.day, this.hour, this.minute, intSeconds, nanos += this.millis * 1000000});
        }

        public TimeStruct copy() {
            TimeStruct result = new TimeStruct();
            result.year = this.year;
            result.month = this.month;
            result.day = this.day;
            result.hour = this.hour;
            result.minute = this.minute;
            result.seconds = this.seconds;
            result.millis = this.millis;
            result.micros = this.micros;
            result.isLocation = this.isLocation;
            return result;
        }

        public TimeStruct add(TimeStruct offset) {
            if (offset.isLocation && this.isLocation) {
                throw new IllegalArgumentException("can't add two times!");
            }
            TimeStruct result = new TimeStruct();
            result.year = this.year + offset.year;
            result.month = this.month + offset.month;
            result.day = this.day + offset.day;
            result.hour = this.hour + offset.hour;
            result.minute = this.minute + offset.minute;
            result.seconds = this.seconds + offset.seconds;
            result.millis = this.millis + offset.millis;
            result.micros = this.micros + offset.micros;
            result.isLocation = this.isLocation || offset.isLocation;
            return result;
        }
    }
}

