#ifndef LIBR_H_EEWEST_20060224
#define LIBR_H_EEWEST_20060224

void getdat_(int * record, int * time);

int timcmp_(int * t1, int * t2);

void timadd_(int * t1, int * ms, int * t2, int * err);

int i4flip_(int * record);


/** Return the calibrated SFR data from a DE PWI data record.
 *
 * This subroutine will read the SFR data from the DE PWI record, calibrate it,
 * and return it in arrays of double precision numbers.
 *
 * @param record - A DE PWI data record.
 *
 * @param recv - The desired receiver, as follows:
 *
 *     RECV      Meaning
 *     ----      ---------
 *      0       Receiver A
 *      1       Receiver B
 *
 * @param ant (output) -- The antenna connected to the desired receiver:
 * 
 *   ANT       Antenna
 *   ---       -------
 *    0          Es
 *    1          Ez
 *    2          Ex
 *    3          B
 *
 * @param sfr -- (output) A 32 by 4 array containing the calibrated SFR data
 *       expressed in the appropriate units of spectral density.
 *
 *
 * @param freq -- (output) A 32 by 4 array cantaining the frequencies of the
 *       data in the SFR arrays expressed in Hertz.
 */
void getsfr_(int record[], int * recv, int * ant, double** sfr, double** freq);


/* Get the calibrated SFR phase data from a DE PWI record.
 *
 * Reads the SFR phase data from the DE PWI record, calibrate it, and
 * return it in arrays of double precision numbers.
 *
 * Arguments:
 *  record (input)  : A DE PWI data record.
 *
 *  antA   (output) : The antenna connected to SFR-A.
 *
 *   ANT       Antenna
 *   ---       -------
 *    0          Es
 *    1          Ez
 *    2          Ex
 *    3          B
 *
 *  antB   (output) : The antenna connected to SFR-B, encoded in the same
 *                    manner as antA.
 *
 *  phase  (output) : A 32 by 4 array containing the calibrated SFR phase
 *                    data expressed in degrees.
 *
 *  corr   (output) : A 32 by 4 array containing the correlation of the
 *                    measured phase.
 *
 *  freq   (output) : A 32 by 4 array contaning the frequencies of the data
 *                    in the phase array expressed in Hertz.
 *
 * Version 1.0  22-Aug-1990  Scott C. Allendorf
 *                           - Original version.
 */
void getphs_(int* record, int* antA, int* antB, double** phase, 
		       double** corr, double** freq);


/** 
 */
void getlfc_(int* record, int* recv, int * ant, double** dataHi,
		       double** dataLo, double* freqHi, double* freqLo);

/** Calibrated LFC phase data from a DE PWI record
 *
 * This subroutine will read the LFC phase data from the DE PWI record,
 * calibrate it, and return it in arrays of double precision numbers.
 *
 * Arguments:
 *  record (input)  : A DE PWI data record, an array of 442 integers.
 *
 *  antA   (output) : The antenna connected to LFC-A.
 *
 *   ANT       Antenna
 *   ---       -------
 *    0          Es
 *    1          Ez
 *    2          Ex
 *    3          H
 *
 *  antB   (output) : The antenna connected to LFC-B, encoded in the same
 *                    manner as antA.
 *
 *  phaseHi(output) : The calibrated LFC high band phase data expressed
 *                    in degrees.  [An Array of 8 by 8 doubles]
 *
 *  phaseLo(output) : The calibrates LFC low band phase data expressed in
 *                    degrees.  [An array of 8 by 8 doubles]
 *
 *  corrHi (output) : The correlation of the measured LFC high band
 *                    phase.  A value of -1.0 indicates that the
 *                    corresponding phase could not be determined. [An
 *                    array of 8 by 8 doubles]
 *
 *  corrLo (output) : The correlation of the measured LFC low band phase.
 *                    A value of -1.0 indicates that the corresponding
 *                    phase could not be determined. [an array of 8 by
 *                    8 doubles]
 *
 *  freqHi (output) : The frequencies of the data in the phaseHi array
 *                    expressed in Hertz. [array of 8 integers]
 *
 *  freqLo (output) : The frequency of the data in the phaseLo array
 *                    expressed in Hertz.
 *
 * Version 1.0  22-Sep-1990  Scott C. Allendorf
 *                           - Original version.
 */
void getphl_(int* record, int* antA, int* antB, double** phaseHi, 
		       double** phaseLo, double** corrHi, double** corrLo,
             double* freqHi, double* freqLo);


/** Calculate spin plane and spin axis DC Electric fields
 *
 * This subroutine will read the DC electric field data from the DE PWI
 * record, calibrate it, remove the effects of the spacecraft spin, and
 * return it in arrays of double precision numbers.  The method used is
 * taken from Dr. Daniel R. Weimer's Master of Science thesis (University
 * of Iowa, 1983).
 *
 * Arguments:
 *
 *  record (input)  : A DE PWI data record.
 *                    Pointer to a 443 element array.
 *
 *  exper  (output) : The component of the electric field measured in the
 *                    spin plane that is perpendicular to the projection
 *                    of the local magnetic field on the spin plane,
 *                    expressed in volts/meter.
 *                    Pointer to a two element array.
 *
 *  expar  (output) : The component of the electric field measured in the
 *                    spin plane that is parallel to the projection of
 *                    the local magnetic field on the spin plane,
 *                    expressed in volts/meter.
 *                    Pointer to a two element array.
 *
 *  exchi  (output) : An indication of the quality of the data taken with
 *                    the Ex antenna.  A value of -1.0 indicates that the
 *                    corresponding EPAR and EPER could not be
 *                    determined.
 *                    Pointer to a two element array.
 *
 *  ezavg  (output) : The average electric field measured by the Ez
 *                    antenna, expressed in volts/meter.
 *                    Pointer to a two element array.
 *
 *  ezsdev (output) : The standard deviation of the electric field
 *                    measured by the Ez antenna.  A value of -1.0
 *                    indicates that the corresponding EZAVE could not be
 *                    determined.
 *                    Pointer to a two element array.
 *
 * Version 1.0  11-Jul-1990  Scott C. Allendorf - Original version.
 * 2013-07-26 BMAG added to SAVE.  Maybe it should be initialized?  LJG
 */
 
void getdc_(int* record, double* exper, double* expar, double* exchi, 
            double* ezavg, double* ezsdev);

/** Return instrument status words from a DE PWI data record.
 *
 * This subroutine will read the instrument status words from the DE PWI
 * record and return it in an array of integer numbers.
 *
 * @param record (input) -- A DE PWI data record, 442 integer array
 *
 * @param status(output) -- The status words, a 27 integer array, will
 *        cantain the following:
 *
 *     Index     Contents
 *     -----     --------
 *      1       LFC Low Band Channel
 *                 0 - 1.78 Hz
 *                 1 - 3.12 Hz
 *                 2 - 6.25 Hz
 *                 3 - 10.0 Hz
 *      2       SFR Skip Mode
 *                 0 - Skip 1
 *                 1 - Skip 8
 *      3       LFC B Antenna
 *                 0 - Es
 *                 1 - Ez
 *                 2 - Ex
 *                 3 - H
 *      4       LFC A Antenna
 *                 0 - Es
 *                 1 - Ez
 *                 2 - Ex
 *                 3 - H
 *      5       SFR B Antenna
 *                 0 - Es
 *                 1 - Ez
 *                 2 - Ex
 *                 3 - B
 *      6       SFR A Antenna
 *                 0 - Es
 *                 1 - Ez
 *                 2 - Ex
 *                 3 - B
 *      7       SFR Gain Select
 *                 0 - Low gain (30 dB attenuator)
 *                 1 - High gain
 *                 2 - Low Gain (30dB attenuator)
 *                 3 - Toggle (switch every 0.5 seconds)
 *      8       CMD 22 Cal Inhibit
 *                 0 - Inhibit
 *                 1 - On
 *      9       SC 17 Cal Enable
 *                 0 - Disable
 *                 1 - Enable
 *     10       SFR Antenna Manual/Automatic
 *                 0 - Manual
 *                 1 - Automatic
 *     11       LFC Antenna Manual/Automatic
 *                 0 - Manual
 *                 1 - Automatic
 *     12       LFC Low Band Lock/Sweep
 *                 0 - Sweep
 *                 1 - Lock
 *     13       LFC High Band Lock/Sweep
 *                 0 - Sweep
 *                 1 - Lock
 *     14       SFR Lock/Sweep
 *                 0 - Sweep
 *                 1 - Lock
 *     15       SFR x4 Rate (T = 0)
 *                 0 - x1
 *                 1 - x4
 *     16       LFC High Band Channel (T = 0)
 *                 0 - 17.8 Hz
 *                 1 - 31.2 Hz
 *                 2 - 62.5 Hz
 *                 3 - 100 Hz
 *     17       SFR Step (T = 0)
 *                 n (0 - 31) SFR frequency step
 *     18       SFR x4 Rate (T = 2)
 *                 0 - x1
 *                 1 - x4
 *     19       LFC High Band Channel (T = 2)
 *                 0 - 17.8 Hz
 *                 1 - 31.2 Hz
 *                 2 - 62.5 Hz
 *                 3 - 100 Hz
 *     20       SFR Step (T = 2)
 *                 n (0 - 31) SFR frequency step
 *     21       SFR x4 Rate (T = 4)
 *                 0 - x1
 *                 1 - x4
 *     22       LFC High Band Channel (T = 4)
 *                 0 - 17.8 Hz
 *                 1 - 31.2 Hz
 *                 2 - 62.5 Hz
 *                 3 - 100 Hz
 *     23       SFR Step (T = 4)
 *                 n (0 - 31) SFR frequency step
 *     24       SFR x4 Rate (T = 6)
 *                 0 - x1
 *                 1 - x4
 *     25       LFC High Band Channel (T = 6)
 *                 0 - 17.8 Hz
 *                 1 - 31.2 Hz
 *                 2 - 62.5 Hz
 *                 3 - 100 Hz
 *     26       SFR Step (T = 6)
 *                 n (0 - 31) SFR frequency step
 *     27       Wide Band Data Present
 *                 0 - No wideband data present
 *                 1 - Wideband data present
 *
 * Version 1.0  15-Feb-1990  Scott C. Allendorf
 */				
void getsts_(int* record, int* status);


/** Return the orbit data from a DE PWI data record.
 *
 * This subroutine will read the orbit data from the DE PWI record and
 * return it in an array of double precision floating point numbers.
 *
 * Arguments:
 *  RECORD (input)  : A DE PWI data record.
 *  ORBIT  (output) : The orbit data, as follows:
 *
 *   Index     Contents                                          Units
 *   -----     ------------------------------------------------  -----
 *     0       GEI satellite velocity vector.              v(x)  km/sec
 *     1 																  v(y)  km/sec
 *     2 																  v(z)  km/sec
 *     3 		Altitude above the spheroid Earth.  				  km
 *     4 		Geodetic latitude of subsatellite point.  		  degrees
 *     5 		East longitude of the satellite. 					  degrees
 *     6 		Local magnetic time. 									  hours
 *     7 		McIlwain's shell parameter (L).  					  Re
 *     8 		Invariant latitude.  									  degrees
 *     9 		Magnetic field strength.								  gauss
 *    10 		GEI magnetic field vector. 					  B(x)  gauss
 *    11 																  B(y)  gauss
 *    12 																  B(z)  gauss
 *    13 		Orbit number.
 *    14 		3-by-3 rotation matrix for the				  X(x)
 *    15 			transformation from spacecraft			  X(y)
 *    16 			coordinates to GEI.  						  X(z)
 *    17 																  Y(x)
 *    18 																  Y(y)
 *    19 																  Y(z)
 *    20 																  Z(x)
 *    21 																  Z(y)
 *    22 																  Z(z)
 *    23 		GEI satellite position vector.				  r(x)  km
 *    24 																  r(y)  km
 *    25 																  r(z)  km
 *    26 		GEI satellite velocity relative to  		  R(x)  km/sec
 *    27 			a rotating atmosphere.  					  R(y)  km/sec
 *    28 																  R(z)  km/sec
 *    29 		GEI unit vector toward the Sun.  			  U(x)  km
 *    30 																  U(y)  km
 *    31 																  U(z)  km
 *    32 		Phase angle of spin measured from the  			  radians
 *                velocity vector to the x-axis of
 *                the spacecraft.
 *    33       Sunlight/Darkness: 0 = Darkness, 1 = Sunlight.
 *    34       Geocentric radial distance (Derived)              Re
 *    35       Magnetic Latitude (Derived)                       degrees
 *
 * Version 1.0  30-Jan-1990  Scott C. Allendorf
 */
void getorb_(int record[], double orbit[]);

/** Calculate the local magnetic field.
 *
 * This subroutine calculates the local magnetic field using the MAGSAT
 * model.  The code was originally derived from the ONEMAG program
 * described in document NSSDC 72-12 from the National Space Science Data
 * Center.  The code was modified to use the MAGSAT coefficients by Dr.
 * Daniel R. Weimer in February 1984.  The secular variation coefficients
 *  are from the IGRF 1980 model.
 *
 * Arguments:
 *  DATE   (input)  : The date expressed as YYDDD
 *                    (January 1 => DDD = 001).
 *  POS    (input)  : The GEI satellite position vector expressed in km.
 *  LAT    (input)  : The geocentric latitude of the spacecraft expressed
 *                    in degrees.
 *  LONG   (input)  : The east longitude of the spacecraft (degrees).
 *  B      (output) : The rectangular GEI components of the local
 *                    magnetic field expressed in gauss.
 *
 *
 * Version 1.0  03-Sep-90  Scott C. Allendorf
 *                        - Modification of MAGSAT subroutine.
 */
void getb_(int* date, double pos[], double* lat, double* lon, double b[]);



#endif /* LIBR_H_EEWEST_20060224 */
