/*
 * VoyagerHighRate.java
 *
 * Created on February 21, 2006, 12:18 PM
 *
 *
 */

package edu.uiowa.physics.pw.apps.vgpws_hr;

//import edu.uiowa.physics.pw.apps.vgpws.*;
import org.das2.DasApplication;
import org.das2.DasException;
import org.das2.components.DasTimeRangeSelector;
import org.das2.components.TearoffTabbedPane;
import org.das2.dataset.DataSetDescriptor;
import org.das2.dataset.DataSetUpdateEvent;
import org.das2.dataset.DataSetUpdateListener;
import org.das2.dataset.DataSetUtil;
import org.das2.dataset.TableDataSet;
import org.das2.dataset.VectorDataSet;
import org.das2.datum.DatumRange;
import org.das2.datum.DatumRangeUtil;
import org.das2.datum.Units;
import org.das2.event.DataPointSelectorMouseModule;
import org.das2.event.VerticalSliceSelectionRenderer;
import org.das2.graph.DasCanvas;
import org.das2.graph.DasColorBar;
import org.das2.graph.DasColumn;
import org.das2.graph.DasPlot;
import org.das2.graph.DasRow;
import org.das2.graph.Legend;
import org.das2.graph.SpectrogramRenderer;
import org.das2.graph.SymbolLineRenderer;
import org.das2.graph.TimeRangeLabel;
import org.das2.util.ArgumentList;
import org.das2.util.DasExceptionHandler;
import org.das2.util.monitor.ProgressMonitor;
import org.das2.util.filesystem.FileSystem;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.event.ActionEvent;
import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JFrame;
import javax.swing.JPanel;
import org.das2.util.FileUtil;
import org.das2.util.filesystem.HttpFileSystem;

/**
 *
 * @author Jeremy
 */
public class VoyagerHighRate {
    DasCanvas canvas;
    SpectrogramRenderer renderer;
    SymbolLineRenderer cycloRenderer;
    SymbolLineRenderer indexedRenderer;
    MagDataSetDescriptor fceDsd;
    IndexTool indexTool;
    
    DasPlot plot;
    HighRateSlicer slicer;
    
    HROptionsPanel optionsPanel;
    
    HashMap overlayRenderers= new HashMap(); // <SymbolLineRenderer>
    HashMap overlayLabels= new HashMap(); // <JLabel>
    
    Legend legend;
    
    LocalFileReaderPanel localReaderPanel;
    
    boolean localOnly= false;
    
    DataSetUpdateListener indexListener= new DataSetUpdateListener() {
        public void dataSetUpdated(DataSetUpdateEvent e) {
            updateIndexed();
        }
    };
    
    protected void updateIndexed() {
        VectorDataSet indexDs= (VectorDataSet)indexTool.getRecorder().getDataSet();
        String voyString= optionsPanel.isVoyager1() ? "1" : "2";
        DataSetDescriptor dsd= new IndexedDensityFileDataSetDescriptor( voyString,
                DensityFileDataSetDescriptor.FCE_PLANE_TYPE, indexDs );
        indexedRenderer.setDataSetDescriptor( dsd );
    }
    
    VoyagerHighRate( boolean localOnly, JFrame frame ) {
        this.localOnly= localOnly;
        
        CanvasPanel canvasPanel= new CanvasPanel();
        optionsPanel= new HROptionsPanel(this);
        localReaderPanel= new LocalFileReaderPanel(this);
        
        TearoffTabbedPane pane= new TearoffTabbedPane();
        
        pane.setOpaque( true );
        pane.addTab( "plot", canvasPanel.getPanel() );
        if ( !localOnly ) pane.addTab( "options", optionsPanel );
        pane.addTab( "local", localReaderPanel );
        
        if ( frame==null ) {
            frame= DasApplication.getDefaultApplication().createMainFrame( "Voyager High Rate", pane );
        } else {
            frame.getContentPane().add(pane);
        }
        
        if ( !localOnly ) {
            indexTool= new IndexTool( plot, frame );
            pane.addTab( "index", indexTool.getRecorder() );
            indexTool.getRecorder().addDataSetUpdateListener(indexListener);
            update();
        } else {
            localReaderPanel.setUseLocal( true );
            renderer.setDataSetLoader(localReaderPanel.loader);
            
        }
        
        
    }
    
    protected void update() {
        try {
            renderer.setDataSetID( optionsPanel.getSpectrogramDataSetID() );
            String dsdClass= "class:edu.uiowa.physics.pw.apps.vgpws.DensityFileDataSetDescriptor";
            if ( optionsPanel.isVoyager1() ) {
                plot.setTitle( "Voyager 1 Wideband" );
            } else {
                plot.setTitle( "Voyager 2 Wideband" );
            }
            fceDsd= new MagDataSetDescriptor( optionsPanel.isVoyager1() ? Spacecraft.voyager1 : Spacecraft.voyager2, true );
            cycloRenderer.setDataSetDescriptor(fceDsd);
        } catch ( DasException e ) {
            DasExceptionHandler.handle(e);
        } catch ( FileSystem.FileSystemOfflineException e ) {
            DasExceptionHandler.handle(e);
        }
    }
    
    float lastHue= -0.07f;
    Color getNextNonColorbarColor() {
        lastHue+= 0.07f;
        return  Color.getHSBColor( lastHue, 0.8f, 0.8f );
    }
    
    class CanvasPanel {
        JPanel panel;
        
        JPanel getPanel() {
            return panel;
        }
        
        CanvasPanel() {
            canvas= new DasCanvas( 800, 600 );
            DasRow row= new DasRow( canvas, null, 0, 1, 4, -4, 0, 0 );
            DasColumn column= new DasColumn( canvas, null, 0, 1, 5, -8, 0, 0 );
            
            DatumRange timeRange= DatumRangeUtil.parseTimeRangeValid("1979-060 12:25:50 to 1979-060 12:28:00" );
            DatumRange freqRange= new DatumRange( 0.04, 12, Units.kiloHertz );
            
            DatumRange ampRange= new DatumRange( 1e-5, 1e-1, Units.dimensionless );
            
            plot= DasPlot.createPlot( timeRange, freqRange );
            plot.getYAxis().setLabel("frequency (kHz)");
            
            canvas.add( plot, row, column );
            
            DasColorBar colorbar= new DasColorBar( ampRange.min(), ampRange.max(), true );
            colorbar.setLabel("relative power spectral density");
            
            renderer= new SpectrogramRenderer( null, colorbar );
            renderer.setActive(false);
            plot.addRenderer( renderer );
            
            
            try {
                fceDsd= new MagDataSetDescriptor(Spacecraft.voyager1,true);
            } catch ( FileSystem.FileSystemOfflineException ex) {
                DasExceptionHandler.handle(ex);
            }
            
            cycloRenderer= new SymbolLineRenderer( );
            cycloRenderer.setDataSetDescriptor(fceDsd);
            cycloRenderer.setActive(false);
            
            indexedRenderer= new SymbolLineRenderer( );
            indexedRenderer.setColor( Color.LIGHT_GRAY );
            indexedRenderer.setSymSize( 1.0 );
            
            indexedRenderer.setActive(false);
            
            plot.addRenderer( renderer );
            plot.addRenderer(cycloRenderer);
            plot.addRenderer(indexedRenderer);
            
            legend= new Legend();
            
            canvas.add( legend, row.createAttachedRow( 0.02, 0.3 ), column.createAttachedColumn( 0.82, 0.99 ) );
            
            slicer= new HighRateSlicer( plot );
            DataPointSelectorMouseModule waveformSlicer= new DataPointSelectorMouseModule( plot, renderer, new VerticalSliceSelectionRenderer( plot ), "waveform" );
            waveformSlicer.addDataPointSelectionListener( slicer );
            
            plot.addMouseModule( waveformSlicer );
            
            if ( localOnly ) {
					/* Remove the slicer from the bottom plot location */
                /* row.setMaximum(0.7);
                row.setEmMaximum(-2);
                DasRow row2= new DasRow( canvas, null, 0.7, 1.0, 3, -3, 0, 0 );
                DasPlot plot2= slicer.createPlot(); 
                slicer.setDisplayer(canvas);
                canvas.add( plot2, row2, column ); */
                renderer.setException( new Exception("(Use local tab to select data files)") );
                renderer.setActive(true);
                plot.getMouseAdapter().setPrimaryModule( waveformSlicer );
            }
            
            canvas.add( new TimeRangeLabel( plot.getXAxis().getDataRange() ), row, column );
            
            panel= new JPanel();
            
            panel.setLayout( new BorderLayout( ) );
            panel.add( canvas, BorderLayout.CENTER  );
            
            if ( !localOnly ) {
                DasTimeRangeSelector tsel= new DasTimeRangeSelector( timeRange );
                panel.add( tsel, BorderLayout.SOUTH );
                
                tsel.addTimeRangeSelectionListener( plot.getXAxis() );
                plot.getXAxis().addTimeRangeSelectionListener( tsel );
            }
            
        }
    }
    
    private Action getUpdateAction() {
        return new AbstractAction( "Update" ) {
            public void actionPerformed( ActionEvent e ) {
                Runnable run= new Runnable() {
                    public void run() {
                        update();
                    }
                };
                new Thread( run ).start();
            }
        };
    }
        
    protected void setLocalFile( final File[] files ) {
        renderer.setActive(true);
        
        if ( renderer.getDataLoader()!=localReaderPanel.loader ) 
			  renderer.setDataSetLoader(localReaderPanel.loader);
        
        Runnable run= new Runnable() {
            public void run() {
                ProgressMonitor monitor= 
								DasApplication.getDefaultApplication().getMonitorFactory().getMonitor( 
								         renderer.getParent(), "Loading data set", "high rate dataloader"
											);
                TableDataSet tds;
                try {
                    tds = localReaderPanel.loader.loadFiles( files, monitor);
                    renderer.setDataSet( tds );
                    renderer.getParent().getXAxis().setDatumRange( DataSetUtil.xRange(tds) );
                    updateLocalFileSC();
                } catch (IOException ex) {
                    ex.printStackTrace();
                    DasExceptionHandler.handle(ex);
                }
            }
        };
        new Thread( run, "load datafile" ).start();
        
    }
    
    private void updateLocalFileSC() {
        if ( localReaderPanel.loader.getFiles()==null ) return;
        int sc= localReaderPanel.loader.getVoyagerSpacecraftNum();
        if ( sc==1 ) {
            plot.setTitle( "Voyager 1 Wideband" );
        } else {
            plot.setTitle( "Voyager 2 Wideband" );
        }
    }

    /**
     * set the application to use local files.  Currently, the application
     * cannot switch back to network access.
     */
    protected void setUseLocal( boolean useLocal ) {
        if ( useLocal ) {
            renderer.setActive(true);
            renderer.setDataSetLoader(localReaderPanel.loader);
            updateLocalFileSC();
        }
    }
    
    public DasPlot getPlot() {
        return plot;
    }
    
    public static void main( String[] args ) {



        final ArgumentList alm= new ArgumentList("voyagerHighrate");
        alm.addBooleanSwitchArgument( "localOnly", "lo", "localOnly", "don't provide server option" );
        alm.process(args);
        
        // delete cache directory, which is causing problems
        File cache= FileSystem.settings().getLocalCacheDir().getAbsoluteFile();
        FileUtil.deleteFileTree(cache);
        
        boolean localOnly= alm.getBooleanValue("localOnly");
        new VoyagerHighRate( localOnly, null );
    }
    
    private boolean addedCycloToLegend= false;
}
