/*
 * VgPdsDSD.java
 *
 * Created on January 3, 2008, 1:41 PM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */
package edu.uiowa.physics.pw.apps.vgpws_hr;

import org.das2.CancelledOperationException;
import org.das2.DasException;
import org.das2.dataset.CacheTag;
import org.das2.dataset.DataSet;
import org.das2.dataset.DataSetDescriptor;
import org.das2.dataset.NoDataInIntervalException;
import org.das2.dataset.VectorDataSetBuilder;
import org.das2.datum.Datum;
import org.das2.datum.Units;
import org.das2.datum.format.TimeDatumFormatter;
import org.das2.util.TimeParser;
import edu.uiowa.physics.pw.pds.PdsField;
import edu.uiowa.physics.pw.pds.PdsFile;
import edu.uiowa.physics.pw.pds.PdsIndexTable;
import edu.uiowa.physics.pw.pds.PdsProduct;
import edu.uiowa.physics.pw.pds.base.PdsRangeException;
import edu.uiowa.physics.pw.pds.PdsSpreadsheet;
import edu.uiowa.physics.pw.pds.PdsVolume;
import edu.uiowa.physics.pw.pds.search.PdsRowList;
import edu.uiowa.physics.pw.pds.search.PdsTblSearcher;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.text.ParseException;
import java.util.List;
import org.das2.util.monitor.ProgressMonitor;
import pds.label.PDSException;
import pds.util.PPITime;

/** DataSet generator for reading DATE and ELECTRON_NUMBER_DENSITY from Vgr Dens PDS.
 *
 * @author cwp
 */
public class VgDensDataDSD extends DataSetDescriptor {

    /** Gather data for voyager 1 or voyager 2 */
    protected String m_sPdsDataSet;
    /** The PdsVolume object from which all data is indirectly extracted
     * usage of the built in index and it's searcher are syncronized. */
    protected PdsVolume g_vol;
    /** Converts time strings in Day of year format to us2000 das datums */
    protected TimeParser m_tpDOY_US2K;
    /** Converts time datums for epoch us2k to DOY PDS strings */
    protected TimeDatumFormatter m_tfUS2K_DOY;
    /** The rowlist that corresponds to items for my particular dataset id */
    protected PdsRowList m_rlDataSet;
    /** The column that we are reading. */
    private String column;

    /** Creates a new instance of VgPdsDSD.
     *
     * @param vol used to access the datasets.  Note that all accesses to the
     * volume object's index and the index's searcher will be syncronized.
     *
     */
    public VgDensDataDSD(PdsVolume vol, int nVgr, DensityFileDataSetDescriptor.PlaneType plane)
            throws DasException, PDSException {

        super("class://vgdenspltr.VgDensDataDSD?root=" + vol.getRootStr() + "&data_set_id=" +
                "VG" + nVgr + "-J-PWS-5-DDR-PLASMA-DENSITY-1S-V1.0");

        m_tpDOY_US2K = TimeParser.create("%Y-%jT%H:%M:%S.%{milli}");
        try {
            m_tfUS2K_DOY = new TimeDatumFormatter("%Y-%jT%H:%M:%S.%{milli}");
        } catch (ParseException ex) {
            throw new DasException(ex);
        }

        if ((nVgr < 1) || (nVgr > 2)) {
            throw new IllegalArgumentException("nVgr must be 1 or 2");
        }
        m_sPdsDataSet = "VG" + nVgr + "-J-PWS-5-DDR-PLASMA-DENSITY-1S-V1.0";
        g_vol = vol;
        try {

            m_rlDataSet = g_vol.getIndexTable().getSearcher().findEq(
                    "DATA_SET_ID", m_sPdsDataSet);
        } catch (IOException ex) {
            throw new DasException(ex);
        }

      if (plane == DensityFileDataSetDescriptor.FCE_PLANE_TYPE) {
            this.column = "FCE";
      } else if (plane == DensityFileDataSetDescriptor.FL_PLANE_TYPE) {
            this.column = "L_FREQ";
      } else if(plane == DensityFileDataSetDescriptor.FR_PLANE_TYPE) {
            this.column = "R_FREQ";
      } else if(plane == DensityFileDataSetDescriptor.FPE_PLANE_TYPE) {
            this.column = "PLASMA_FREQ";
      } else if(plane == DensityFileDataSetDescriptor.FUH_PLANE_TYPE ) {
            this.column = "UH_FREQ";
      } 

    }

    public int getVgr() {
        if (m_sPdsDataSet.startsWith("VG1")) {
            return 1;
        } else {
            return 2;
        }
    }

    /** Given a time range, produce a data set.
     * Asks the contained PdsVolume object for all products in the indicated time
     * then looks through those products for a all data in range and converts to
     * a Das DataSet.
     */
    protected DataSet getDataSetImpl(Datum start, Datum end, Datum resolution,
            ProgressMonitor monitor) throws DasException {

        // Get time in PDS Day of Year format.
        String sStart = m_tfUS2K_DOY.format(start, Units.us2000);
        String sEnd = m_tfUS2K_DOY.format(end, Units.us2000);
        VectorDataSetBuilder builder =
                new VectorDataSetBuilder(Units.us2000, Units.hertz);

        //if(monitor instanceof DasProgressPanel)
        //	((DasProgressPanel)monitor).setShowProgressRate(false);

        if (monitor.isCancelled()) {
            throw new CancelledOperationException();
        }
        monitor.setLabel("Loading Data");

        if (monitor.isCancelled()) {
            throw new CancelledOperationException();
        }
        monitor.setProgressMessage("Finding Products");

        if (monitor.isCancelled()) {
            throw new CancelledOperationException();
        }
        monitor.started();

        try {
            PPITime tStart = new PPITime();
            if (!tStart.convert(PPITime.PDS, sStart)) {
                throw new IllegalArgumentException("Could not convert " +
                        sStart + " to a timestamp.");
            }
            PPITime tEnd = new PPITime();
            if (!tEnd.convert(PPITime.PDS, sEnd)) {
                throw new IllegalArgumentException("Could not convert " +
                        sEnd + " to a timestamp.");
            }
            PdsIndexTable idx = g_vol.getIndexTable();
            List<String> prodAry = null;

            synchronized (idx) {
                PdsTblSearcher searcher = idx.getSearcher();
                PdsRowList rl, rlNext;

                //If begin time in bounds
                rl = searcher.findGtEq("START_TIME", tStart).AND(
                        searcher.findLt("START_TIME", tEnd));

                //If end time in bounds
                rlNext = searcher.findGt("STOP_TIME", tStart).AND(
                        searcher.findLtEq("STOP_TIME", tEnd));
                rl.OR(rlNext);

                //if begin and end encompass the bounds
                rlNext = searcher.findLt("START_TIME", tStart).AND(
                        searcher.findGtEq("STOP_TIME", tEnd));
                rl.OR(rlNext);

                rl.AND(m_rlDataSet);

                prodAry = idx.getProducts(rl);
            }

            if (prodAry.isEmpty()) {
                throw new NoDataInIntervalException("");
            }
            if (monitor.isCancelled()) {
                throw new CancelledOperationException();
            }
            monitor.setTaskSize(prodAry.size() * 100);

            int iSheet = -1;
            for (String sRelPath : prodAry) {
                monitor.setProgressMessage("Reading " + sRelPath);
                iSheet += 1;

                if (monitor.isCancelled()) {
                    throw new CancelledOperationException();
                }
                monitor.setTaskProgress(iSheet * 100);

                PdsProduct prod = g_vol.getProduct(sRelPath);
                PdsFile file = prod.getFile();
                PdsSpreadsheet sheet = file.subSpreadsheet("DENSITY");
                PdsField fldDate = sheet.subField("SCET");
                PdsField fldDens = sheet.subField( column );

                //I know that dates are in DOY format, so optimize using a string search
                //Furthermore I know that the dates are monotonically increasing, so
                //the first one that is too large stops the loop.
                int nRowFrac = 100 / sheet.size();
                for (int i = 0; i < sheet.size(); i++) {
                    sheet.setRow(i);
                    String sDate = fldDate.getString();

                    if (sDate.compareTo(sEnd) > 0) {
                        break;
                    }
                    if (sDate.compareTo(sStart) >= 0) {

                        if (fldDens.getItemSize() > 0) {
                            double dDens = fldDens.doubleVal();

                            double dTime = m_tpDOY_US2K.parse(sDate).getTime(Units.us2000);
                            Units.us2000.createDatum(dTime);
                            builder.insertY(dTime, dDens);
                        }
                    }

                    if (monitor.isCancelled()) {
                        throw new CancelledOperationException();
                    }
                    monitor.setTaskProgress(iSheet * 100 + (i + 1) * nRowFrac);
                }
            }
        } catch (PDSException ex) {
            throw new DasException(ex);
        } catch (FileNotFoundException ex) {
            throw new DasException(ex);
        } catch (IOException ex) {
            throw new DasException(ex);
        } catch (ParseException pe) {
            throw new DasException(pe);
        } catch (PdsRangeException ex) {
            throw new DasException(ex);
        }

        builder.setProperty(DataSet.PROPERTY_CACHE_TAG, new CacheTag(start, end, null));
        builder.setProperty(DataSet.PROPERTY_X_TAG_WIDTH, Units.seconds.createDatum(20));
        //builder.setProperty()

        if (monitor.isCancelled()) {
            throw new CancelledOperationException();
        }
        monitor.finished();

        return builder.toVectorDataSet();
    }

    public Units getXUnits() {
        return Units.us2000;
    }

    public void requestDataSet(final Datum start, final Datum end, final Datum resolution,
            final ProgressMonitor monitor, Object lockObject) {
        super.requestDataSet(start, end, resolution, monitor, lockObject);
    }
}
