/*
 * HighRateSlicer.java
 *
 * Created on February 27, 2006, 4:24 PM
 *
 *
 */

package edu.uiowa.physics.pw.apps.vgpws_hr;

import org.das2.DasException;
import org.das2.client.StreamDataSetDescriptor;
import org.das2.client.WebStandardDataStreamSource;
import org.das2.components.ComponentsUtil;
import org.das2.dataset.DataSetDescriptor;
import org.das2.dataset.DataSetUpdateEvent;
import org.das2.dataset.DataSetUpdateListener;
import org.das2.dataset.DataSetUtil;
import org.das2.dataset.TableDataSet;
import org.das2.dataset.VectorDataSet;
import org.das2.datum.Datum;
import org.das2.datum.DatumRange;
import org.das2.datum.TimeUtil;
import org.das2.datum.Units;
import org.das2.event.DataPointSelectionEvent;
import org.das2.event.DataPointSelectionListener;
import org.das2.graph.DasAxis;
import org.das2.graph.DasCanvas;
import org.das2.graph.DasColumn;
import org.das2.graph.DasPlot;
import org.das2.graph.DasRow;
import org.das2.graph.DataLoader;
import org.das2.graph.SymbolLineRenderer;
import org.das2.graph.XAxisDataLoader;
import org.das2.util.DasExceptionHandler;
import org.das2.util.monitor.ProgressMonitor;
import java.io.File;
import java.io.IOException;
import java.text.DecimalFormat;
import javax.swing.JDialog;
import javax.swing.SwingUtilities;

/**
 *
 * @author Jeremy
 */
public class HighRateSlicer implements DataPointSelectionListener {
    
    SymbolLineRenderer waveformRenderer;
    SymbolLineRenderer fftRenderer;
    DasPlot parentPlot;
    DasPlot plot1;
    
    JDialog popup= null;
    boolean popupLoaded= false;
    
    DasAxis xaxis;
    DataLoader parentLoader;
    Strategy strategy;
    
    interface Strategy {
        void getWaveDataSet( Datum time );
        void describeDataSet( );
        void nextRecord();
        void previousRecord();
    }
    
    class Das2ServerStrategy implements Strategy {
        
        StreamDataSetDescriptor waveDsd=null;
        TableDataSet waveTable=null;
        Exception loadException=null;
        DatumRange loadedRange= null;
        DataSetUpdateListener listener;
        Datum pendingTime=null;
        int currentRecordIndex;
        String parentDsId;
        
        Das2ServerStrategy( XAxisDataLoader loader ) {
            parentDsId= loader.getDataSetDescriptor().getDataSetID();
            resetDataSetDescriptor();
        }
        
        private void resetDataSetDescriptor() {
            int i= parentDsId.indexOf( "pws/wf-" );
            String waveId= parentDsId.substring(0,i+"pws/wf-".length())+"waveform2";
            try {
                waveDsd= (StreamDataSetDescriptor) DataSetDescriptor.create(waveId);
                ((WebStandardDataStreamSource)waveDsd.getStandardDataStreamSource()).setCompress(true);
                ((WebStandardDataStreamSource)waveDsd.getStandardDataStreamSource()).setDevel("jbf");
            } catch (DasException ex) {
                throw new RuntimeException(ex);
            }
        }
        
        public synchronized void getWaveDataSet( final Datum time) {
            if ( parentDsId!=((XAxisDataLoader)parentLoader).getDataSetDescriptor().getDataSetID() ) {
                parentDsId= ((XAxisDataLoader)parentLoader).getDataSetDescriptor().getDataSetID();
                resetDataSetDescriptor();
            }
            if ( loadedRange==null || !( loadedRange.contains( time )|| loadedRange.max().equals(time) ) ) {
                waveTable= null;
                if ( pendingTime!= null ) {
                    pendingTime= time;
                    return;
                } else {
                    listener= new DataSetUpdateListener() {
                        public void dataSetUpdated(DataSetUpdateEvent e) {
                            waveTable= (TableDataSet) e.getDataSet();
                            if ( waveTable==null ) {
                                loadException= e.getException();
                                pendingTime=null;
                                waveformRenderer.setException(loadException);
                                waveformRenderer.setDataSet(null);
                            } else {
                                loadedRange= DataSetUtil.xRange(waveTable);
                                loadedRange.include( pendingTime ); // kludge for contains
                                getWaveDataSet( pendingTime );
                                describeDataSet();
                                pendingTime=null;
                            }
                        }
                    };
                    pendingTime= time;
                    DatumRange range= new DatumRange(time,time.add(1,Units.milliseconds));
                    ProgressMonitor mon= plot1.getCanvas().getApplication().getMonitorFactory().getMonitor( plot1, "loading waveform data", "loading waveform data "+range );
                    waveDsd.requestDataSet( range.min(), range.max(), null, mon, plot1.getCanvas(), listener );
                    return;
                }
            }
            if ( waveTable!=null ) {
                int i= DataSetUtil.closestColumn( waveTable, time );
                currentRecordIndex= i;
                VectorDataSet vds= waveTable.getXSlice( i );
                waveformRenderer.setDataSet( vds );
            } else {
                waveformRenderer.setException(loadException);
                waveformRenderer.setDataSet(null);
            }
        }
        
        public void describeDataSet() {
            if ( waveTable==null ) {
                plot1.setTitle("loading");
            } else {
                Datum time= TimeUtil.createValid((String)waveTable.getProperty("scet"));
                Datum offset= waveTable.getXTagDatum(currentRecordIndex).subtract(time);
                int line= (int)Math.round( offset.doubleValue(Units.milliseconds) / 60 );
                DecimalFormat df= new DecimalFormat("000");
                
                
                String d= "das2Server: "+waveTable.getProperty("spacecraft")+" "+waveTable.getProperty("instrument_id")
                        +" "+waveTable.getProperty("sclk")+":"+df.format(line)+" waveform"
                        + "!c"+waveTable.getXTagDatum(currentRecordIndex);
                plot1.setTitle(d);
            }
        }
        
        public void nextRecord() {
            currentRecordIndex++;
            if ( currentRecordIndex == waveTable.getXLength() ) {
                currentRecordIndex= waveTable.getXLength()-1;
            }
            VectorDataSet vds= waveTable.getXSlice( currentRecordIndex );
            waveformRenderer.setDataSet( vds );
            describeDataSet();
            
        }
        
        public void previousRecord() {
            currentRecordIndex--;
            if ( currentRecordIndex==-1 ) {
                currentRecordIndex=0;
            }
            VectorDataSet vds= waveTable.getXSlice( currentRecordIndex );
            waveformRenderer.setDataSet( vds );
            describeDataSet();
        }
        
    }
    
    class LocalHRFileStrategy implements Strategy {
        //URL root= "file://B:/voyager/web/data//data/VGPW_1004/EXTRAS/V10403.HTM";
        File file= null;
        DatumRange fileTimeRange=null;
        HighRateReader reader=null;
        int line;
        int currentRecordIndex;
        
        /**
         * loader responsible for getting the data that keeps track of the current, loaded file.
         */
        HighRateFilesDataLoader hrLoader;
        
        LocalHRFileStrategy( HighRateFilesDataLoader hrLoader ) {
            this.hrLoader= hrLoader;
        }
        
        private void updateFile( Datum time ) {
            try {
                File file= hrLoader.getFileContaining(time);
                if ( file==null ) {
                    reader=null;
                    fileTimeRange= null;
                } else {
                    fileTimeRange= hrLoader.getRangeForFile(file);
                    reader= new HighRateReader( file );
                }
            } catch (IOException ex) {
                DasExceptionHandler.handle(ex);
            }
        }
        
        public void getWaveDataSet(Datum time) {
            if ( reader==null || !fileTimeRange.contains(time) ) updateFile( time );
            if ( reader==null ) return;
            int line= reader.getLineContainingTime( time.add( Units.milliseconds.createDatum(30) ) );
            currentRecordIndex= line;
            VectorDataSet lineDs= reader.readRecord( line );
            waveformRenderer.setDataSet( lineDs );
        }
        
        public void describeDataSet() {
            if ( reader==null ) {
                plot1.setTitle("reader is null, switch frames");
                return;
            }
            DecimalFormat df= new DecimalFormat("000");
            plot1.setTitle( " "+reader.getFileTitle()+":"+df.format(reader.getLineCounter(currentRecordIndex))+" waveform" );
        }
        
        public void nextRecord() {
            currentRecordIndex++;
            if ( currentRecordIndex == reader.recordCount() ) {
                currentRecordIndex= reader.recordCount()-1;
            }
            VectorDataSet vds= reader.readRecord(currentRecordIndex);
            waveformRenderer.setDataSet( vds );
            describeDataSet();
            
        }
        
        public void previousRecord() {
            currentRecordIndex--;
            if ( currentRecordIndex==-1 ) {
                currentRecordIndex=0;
            }
            VectorDataSet vds= reader.readRecord( currentRecordIndex );
            waveformRenderer.setDataSet( vds );
            describeDataSet();
        }
        
    }
    
    int currentRecordIndex=0;
    
    
    HighRateSlicer( DasPlot parent ) {
        this.parentPlot= parent;
    }
    
    
    public DasPlot createPlot(  ) {
        DatumRange xdr= new DatumRange( 0, 60, Units.milliseconds );
        DatumRange ydr= new DatumRange( -8, 8, Units.dimensionless );
        
        xaxis= new DasAxis( xdr, DasAxis.HORIZONTAL ) {
            public void scanPrevious() {
                strategy.previousRecord();
            }
            public void scanNext() {
                strategy.nextRecord();
            }
        };
        DasAxis yaxis= new DasAxis( ydr, DasAxis.VERTICAL );
        
        plot1= new DasPlot( xaxis, yaxis );
        
        xaxis= plot1.getXAxis();
        
        xaxis.setAnimated(false);
        plot1.getYAxis().setLabel("4 bit waveform");
        
        waveformRenderer= new SymbolLineRenderer( );
        waveformRenderer.setLineWidth(1.0f);
        
        waveformRenderer.setException( new Exception("(Click on the spectrogram for waveform)") );
        
        plot1.addRenderer( waveformRenderer );
        return plot1;
    }
    
    private void createPopup() {
        DasCanvas canvas= ComponentsUtil.createPopupCanvas( parentPlot, "Waveform Slicer", 800, 200 );
        
        popup= (JDialog)SwingUtilities.getWindowAncestor(canvas);
        
        DasRow row1= new DasRow( canvas, null, 0, 1, +3, -3, 0, 0 );
        
        createPlot();
        
        DasColumn col= new DasColumn( canvas, null, 0, 1, 6, -3, 0, 0 );
        
        canvas.add( plot1, row1, col );

        popupLoaded= true;
        
    }
    
    private void showPopup() {
        if ( popupLoaded ) return;
        if ( popup==null ) createPopup();
        popup.setVisible(true);
    }

    public void setDisplayer( DasCanvas canvas ) {
        popupLoaded= true;
    }

    private boolean isPopupVisible() {
        return popup!=null && popup.isVisible();
    }
    
    public void dataPointSelected(DataPointSelectionEvent e) {
        DataLoader loader= parentPlot.getRenderer(0).getDataLoader();
        if ( loader!=parentLoader ) {
            if ( loader instanceof HighRateFilesDataLoader ) {
                strategy= new LocalHRFileStrategy( (HighRateFilesDataLoader) loader );
            } else if ( loader instanceof XAxisDataLoader ) {
                XAxisDataLoader xloader= (XAxisDataLoader)loader;
                strategy= new Das2ServerStrategy( xloader );
            }
            parentLoader= loader;
        }
        
        showPopup();
        
        strategy.getWaveDataSet( e.getX() );
        
        strategy.describeDataSet( );
        
    }
    
    void setLoader(HighRateFilesDataLoader loader) {
        //this.hrLoader= loader;
    }
}
