/*
 * HighRateReader.java
 *
 * Created on February 24, 2006, 4:41 PM
 *
 *
 */

package edu.uiowa.physics.pw.apps.vgpws_hr;

import org.das2.dataset.VectorDataSet;
import org.das2.datum.Datum;
import org.das2.datum.TimeUtil;
import org.das2.datum.Units;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.FileChannel.MapMode;

/**
 *
 * @author Jeremy
 */
public class HighRateReader {
    
    private ByteBuffer buf;
    private String fileTitle;
    private Datum baseTime;    
    private int recordCount;
    private File file;
    
    public HighRateReader( File f ) throws IOException {
        openFile( f );
    }
    
    private void openFile( File f ) throws IOException {
        FileChannel fc= new FileInputStream( f ).getChannel();
        long fileSize= fc.size();
        buf= fc.map( MapMode.READ_ONLY, 0, fileSize );
        recordCount= (int)(fileSize/1024)-1;
        byte[] titleBytes= new byte[1024-248];
        buf.position(248);
        buf.limit( 248 + titleBytes.length );
        buf.get( titleBytes );
        buf.position(0);
        buf.limit( buf.capacity() );
        fileTitle= new String( titleBytes );
        int idxNull= fileTitle.indexOf('\u0000');
        fileTitle= fileTitle.substring(0,idxNull);
        baseTime= TimeUtil.createValid(fileTitle.substring(26));
        file= f;
    }
    
    public class LineVectorDataSet extends EZVectorDataSet {
        double [] saveData= new double[1600];
        double baseUs2000;
        /**
         * @param line the line counter value of the record
         * @param recbuf the data itsself
         * @param baseTime the base time for the file.
         */
        LineVectorDataSet( ByteBuffer recbuf, Datum baseTime ) {
            int line = ( ((int) recbuf.get(22) & 0xFF) << 8) | ((int)recbuf.get(23) & 0xFF);
            for (int i = 220; i<220+800; i++) {
                double sample = ( ((int) recbuf.get(i) & 0xF0 ) >>>4 ) & 0x0F;
                saveData[(i-220)*2] = sample - 7.5 ;
                sample = (int)recbuf.get(i) & 0x0F;
                saveData[(i-220)*2+1] = sample - 7.5;
            }
            baseUs2000= baseTime.doubleValue( Units.us2000 ) + (line-1) * 60000;
            properties.put( "baseTime", Units.us2000.createDatum(baseUs2000) );
        }
        
        public double getDouble(int i, Units units) {
            return Units.dimensionless.convertDoubleTo(units,saveData[i]);
        }
        
        public int getXLength() {
            return 1600;
        }
        
        public double getXTagDouble(int i, Units units) {
            return Units.microseconds.convertDoubleTo( units, i * 1e6 / 28800. );
        }
        
        public Units getXUnits() {
            return Units.microseconds;
        }
        
        public Units getYUnits() {
            return Units.dimensionless;
        }
    }
    
    /**
     * @return the line counter for the record.  1 is the first record.
     */
    protected int getLineCounter( int record ) {
        return ( ((int) buf.get((record+1)*1024+22) & 0xFF) << 8) | ((int)buf.get((record+1)*1024+23) & 0xFF);
    }
    
    /**
     * @return the index of the record containing the time.  Note 0 is the first record 
     * in the file.
     */
    public int getLineContainingTime( Datum time ) {
        double msDiff= time.subtract( baseTime ).doubleValue( Units.milliseconds );
        int line= (int)( msDiff / 60.000 ) + 1; // 1=first record.
        line= Math.max( 1, Math.min( line,recordCount() ) );
        int result= line-1;        
        while ( result < recordCount() && getLineCounter( result ) < line ) result++;
        while ( result > 1 && getLineCounter( result ) > line ) result--;
        return result;
    }
    
    /**
     * @return the number of records in the file.
     */
    public int recordCount() {
        return recordCount;
    }
    
    /**
     * reads a record from the file.  A VectorDataSet with 1600 samples is returned,
     * @param recordNumber the index of the record to read.  0 is the first record, recordCount-1 is the last.
     * @return a VectorDataSet containing the waveform data.
     */ 
    public VectorDataSet readRecord( int recordNumber ) {
        try {
            buf.limit( 1024 * ( recordNumber+2 ) );
            buf.position( 1024 * ( recordNumber+1 ) );
            ByteBuffer recBuffer= buf.slice();
            buf.position(0);
            buf.limit( 1024 * (recordCount+1) );
            VectorDataSet result= new LineVectorDataSet( recBuffer, baseTime );
            return result;
        } catch ( IllegalArgumentException e ) {
            throw new IllegalArgumentException( "Bad line: "+recordNumber+" file size="+buf.capacity() );
        }
    }
    
    public String getFileTitle() {
        return fileTitle.substring(0,25);
    }
    
    /**
     * @return the base time of the record as identified in
     * bytes 248-1024 of the header.
     */
    public Datum getBaseTime() {
        return this.baseTime;
    }
}

