/*
 * HighRateFilesDataLoader.java
 *
 * Created on February 28, 2006, 1:28 PM
 *
 *
 */

package edu.uiowa.physics.pw.apps.vgpws_hr;

import org.das2.components.DasProgressPanel;
import org.das2.components.propertyeditor.Enumeration;
import org.das2.dataset.ClippedVectorDataSet;
import org.das2.dataset.DataSet;
import org.das2.dataset.DataSetUtil;
import org.das2.dataset.TableDataSet;
import org.das2.dataset.TableDataSetBuilder;
import org.das2.dataset.VectorDataSet;
import org.das2.datum.Datum;
import org.das2.datum.DatumRange;
import org.das2.datum.DatumVector;
import org.das2.datum.Units;
import org.das2.graph.DataLoader;
import org.das2.graph.Renderer;
import org.das2.math.fft.FFTUtil;
import org.das2.math.fft.GeneralFFT;
import org.das2.system.DasLogger;
import org.das2.util.DasExceptionHandler;
import org.das2.util.monitor.ProgressMonitor;
import java.io.File;
import java.io.IOException;
import java.util.logging.Logger;
import javax.swing.Icon;

/**
 *
 * @author Jeremy
 */
public class HighRateFilesDataLoader extends DataLoader {
    static Logger logger= DasLogger.getLogger();
    File[] files;
    DatumRange[] fileRanges;
    
    // 2000-12-12 14:51:54.455 to 14:52:42.395
    
    /**
     * pluggable method for fft and average
     */
    FftAvg fftAvg;
    
    static abstract class FftAvg implements Enumeration {
        GeneralFFT fft;
        public static final FftAvg fft512x3= new Fft512x3();
        public static final FftAvg fft512x3av= new Fft512x3av();
        public static final FftAvg fft1600= new Fft1600();
        DatumVector ytags;
        VectorDataSet weights;
        abstract void doFftAvg( VectorDataSet vds, TableDataSetBuilder builder );
        abstract String getListLabel();
        public Icon getListIcon() {
            return null;
        }
    }
    
    final static class Fft512x3 extends FftAvg {
        Fft512x3() {
            fft= new GeneralFFT( 512, true, true );
            weights= FFTUtil.getWindow10PercentEdgeCosine(512);
        }
        public void doFftAvg(VectorDataSet ds, TableDataSetBuilder builder) {
            int ii=64;
            VectorDataSet powDs= FFTUtil.fftPower( fft, new ClippedVectorDataSet( ds, ii, 512 ), weights );
            if ( ytags==null ) ytags= getXTags(powDs);
            builder.insertYScan( ds.getXTagDatum(ii), ytags, getYValues( powDs ) );
            ii+=512;
            powDs= FFTUtil.fftPower( fft, new ClippedVectorDataSet( ds, ii, 512 ), weights );
            builder.insertYScan( ds.getXTagDatum(ii), ytags, getYValues( powDs ) );
            ii+=512;
            powDs= FFTUtil.fftPower( fft, new ClippedVectorDataSet( ds, ii, 512 ), weights );
            builder.insertYScan( ds.getXTagDatum(ii), ytags, getYValues( powDs ) );
            
        }
        String getListLabel() {
            return "FFT 512x3";
        }
    }
    
    final static class Fft512x3av extends FftAvg {
        double[] s;
        Fft512x3av() {
            fft= new GeneralFFT( 512, true, true );
            weights= FFTUtil.getWindow10PercentEdgeCosine(512);
            s= new double[256];
        }
        
        public void doFftAvg(VectorDataSet ds, TableDataSetBuilder builder) {
            int ii=64;
            for ( int i=0;i<256; i++ ) s[0]=0.0;
            
            VectorDataSet powDs1= FFTUtil.fftPower( fft, new ClippedVectorDataSet( ds, ii, 512 ), weights );
            VectorDataSet powDs2= FFTUtil.fftPower( fft, new ClippedVectorDataSet( ds, ii+=512, 512 ), weights );
            VectorDataSet powDs3= FFTUtil.fftPower( fft, new ClippedVectorDataSet( ds, ii+=512, 512 ), weights );
            
            if ( ytags==null ) ytags= getXTags(powDs1);
            
            for ( int i=0;i<256; i++ ) {
                s[i]= powDs1.getDouble(i,Units.dimensionless)
                + powDs2.getDouble(i,Units.dimensionless)
                + powDs3.getDouble(i,Units.dimensionless);
                s[i]/=3;
            }
            
            builder.insertYScan( (Datum) ds.getProperty("baseTime"), ytags,
                    DatumVector.newDatumVector(s,Units.dimensionless) );
            
        }
        
        String getListLabel() {
            return "FFT 512x3 avg";
        }
    }
    
    final static class Fft1600 extends FftAvg {
        Fft1600() {
            fft= new GeneralFFT( 1600, true, true );
            weights= FFTUtil.getWindow10PercentEdgeCosine(1600);
        }
        public void doFftAvg(VectorDataSet ds, TableDataSetBuilder builder) {
            VectorDataSet powDs= FFTUtil.fftPower( fft, ds, weights );
            if ( ytags==null ) ytags= getXTags(powDs);
            builder.insertYScan( ds.getXTagDatum(0), ytags, getYValues( powDs ) );
        }
        
        String getListLabel() {
            return "FFT 1600";
        }
    }
    
    public void setFftMethod( FftAvg fftAvg) {
        this.fftAvg= fftAvg;
    }
    
    public FftAvg getFftMethod() {
        return this.fftAvg;
    }
        
    public HighRateFilesDataLoader( Renderer r ) {
        super( r );
        fftAvg= fftAvg.fft512x3av;
    }
    
    public void update() {
    }
    
    public void doLoad() {
        if ( this.files==null ) return;
        Runnable run= new Runnable() {
            public void run() {
                try {
                    TableDataSet tds= loadFiles(HighRateFilesDataLoader.this.files,getMonitor("reading high rate data"));
                    getRenderer().setDataSet( tds );
                    getRenderer().getParent().getXAxis().setDatumRange( DataSetUtil.xRange(tds) );
                } catch ( IOException ex ) {
                    DasExceptionHandler.handle(ex);
                }
            }
        };
        new Thread(run,"loadHighRateFiles").start();
    }
    
    private static DatumVector getXTags( DataSet ds ) {
        Units u= ds.getXUnits();
        double[] d= new double[ds.getXLength()];
        for ( int i=0; i<d.length; i++ ) d[i]= ds.getXTagDouble(i,u);
        return DatumVector.newDatumVector(d,u);
    }
    
    private static DatumVector getYValues( VectorDataSet ds ) {
        Units u= ds.getYUnits();
        double[] d= new double[ds.getXLength()];
        for ( int i=0; i<d.length; i++ ) d[i]= ds.getDouble(i,u);
        return DatumVector.newDatumVector(d,u);
    }
    
    public TableDataSet loadFiles( File[] files, ProgressMonitor mon ) throws IOException {
        
        Datum xTagWidth=  Units.milliseconds.createDatum(60);
        
        DatumRange[] fileRanges_= new DatumRange[ files.length ];
        
        TableDataSetBuilder builder= new TableDataSetBuilder( Units.us2000, Units.kiloHertz, Units.dimensionless );
        
        builder.setProperty( TableDataSet.PROPERTY_X_TAG_WIDTH, xTagWidth );
        DatumVector ytags= null;
        
        mon.setTaskSize( files.length * 800 );
        mon.started();
        for ( int i=0; i<files.length; i++ ) {
            HighRateReader reader= new HighRateReader( files[i] );
            Datum min=null, max=null;
            for ( int j=0; j<reader.recordCount(); j++ ) {
                VectorDataSet ds= reader.readRecord( j );
                if ( j==0 ) {
                    min= (Datum) ds.getProperty( "baseTime" );
                } else if ( j==reader.recordCount()-1) {
                    max= ((Datum) ds.getProperty( "baseTime" )).add( xTagWidth );
                }
                
                boolean nonZero= ds.getDouble(0,Units.dimensionless ) > 0.;
                for ( int k=0; !nonZero && k<1600; k++ ) {
                    if ( ds.getDouble(k,Units.dimensionless ) > 0. ) nonZero=true;
                }
                
                if ( nonZero ) {
                    fftAvg.doFftAvg( ds, builder );
                }
                
                // Read each file, appending dataset as we go along.
                //  renderer.setDataSet(ds)
                //  maybe keep track of files for the slicer.
                mon.setTaskProgress( i * 800 + j );
                Thread.currentThread().yield();
                
                if ( mon.isCancelled() ) break;
            }
            fileRanges_[i]= new DatumRange( min, max );
        }
        mon.finished();
        TableDataSet spectrum= builder.toTableDataSet();
        logger.info( "loaded "+DataSetUtil.xRange(spectrum) );
        this.fileRanges= fileRanges_;
        this.files= files;
        return spectrum;
    }
        
    public void setFiles( File[] files ) {
        this.files= files;
    }
    
    /**
     * Return the File that was use to create the spectrum containing @param time, or null
     * if no such file exists.
     */
    public File getFileContaining( Datum time ) {
        if ( fileRanges==null ) return null;
        File result= null;
        for ( int i=0; i<fileRanges.length; i++ ) {
            if ( fileRanges[i].contains(time) ) {
                result= files[i];
                break;
            }
        }
        return result;
    }
    
    public DatumRange getRangeForFile( File file ) {
        if ( fileRanges==null ) return null;
        DatumRange result= null;
        for ( int i=0; i<fileRanges.length; i++ ) {
            if ( files[i].equals(file) ) {
                result= fileRanges[i];
                break;
            }
        }
        return result;
    }
    
    public static void main( String[] args ) throws Exception {
        long t0= System.currentTimeMillis();
        HighRateFilesDataLoader loader= new HighRateFilesDataLoader(null);
        File[] f= new File[] { new File( "B:/voyager/web/data/VGPW_2005/DATA/WFRM/P2/V20503/C4407635.DAT" ) } ;
        TableDataSet tds= loader.loadFiles(f,DasProgressPanel.createFramed("reading high rate files"));
        System.err.println(tds);
        System.err.println( DataSetUtil.xRange(tds));
        System.err.println( "load took "+(System.currentTimeMillis()-t0)+" millis" );
    }
    
    int getVoyagerSpacecraftNum() {
        String f= files[0].toString();
        String snum= ""+f.charAt( f.length()-18 );
        return Integer.parseInt(snum);
    }

    protected File[] getFiles() {
        return files;
    }
}
