; vgpws_wf_plot.pro
;
; Example IDL code to extract Voyager PWS waveform samples
; and plot the waveform and relative power spectrum of each of the lines
; of data in a given file.  See 'C' example in SOFTWARE/C for more
; information.

; 4-bit waveform samples can be re-mapped to floating point -7.5 to 7.5
table = findgen(16) - 7.5

; 1600 samples in time at 28,800 samples per second
time = findgen(1600) / 28.8 ; in ms

; 801 frequency components will be produced by the FFT
; The highest frequency will be the Nyquist frequency (half the sample
; rate)
freq = findgen(801) / 800. * 14.4 ; in kHz

; Input file name
fname = '../C/TESTIN.DAT'
read, 'Enter input file name: ', fname
openr, 1, fname

; 1024-byte records
rec = bytarr(1024)

; Read the first (file header) record and extract the PWS ASCII label
; info.
readu, 1, rec
label = string(rec(248:298))

; There will be 1600 4-bit waveform samples per record
samples = intarr(1600)

; As an IDL trick, we'll use an array of the even indices in order
; to extract first the high 4-bit values and then the low 4-bit values
even = indgen(800) * 2

; Loop until there's a read error (usually 800 lines)
on_ioerror, done
while (1) do begin

  readu, 1, rec

  ; Extract the SCLK line count from bytes 23 and 24 (MSB 16-bit
  ; integer)
  line = fix(rec(22)) * 256 + fix(rec(23))

  ; The time offset relative to the time in the label, in seconds
  offset = 0.06 * (line - 1)

  ; Extract the high-order 4-bit samples
  samples(even) = fix(rec(220:1019)) / 16

  ; Then the interlaced low-order 4-bit samples
  samples(even+1) = fix(rec(220:1019)) mod 16

  ; Two plots per page
  !p.multi = [0, 1, 2]

  ; Top plot is the waveform
  plot, time, samples, $
    yrange=[0, 15], ystyle=1, $
    xtitle="relative time [ms]", ytitle="amplitude", $
    title=label + string(offset, format='(" +", f6.2, " sec")')

  ; Re-map the samples, multiply by a Hamming window, and transform.
  ; (Removing hanning() below will introduce noise due to the
  ; discontinuity between the ends of the sample window and will also
  ; affect the overall power amplitude, but is reasonable for quick-and-
  ; dirty analysis.)
  wfrm = table(samples) * hanning(n_elements(samples), alpha=0.54)
  spectrum = abs(fft(wfrm))

  ; Bottom plot is the (uncalibrated) power spectrum
  plot, freq, spectrum(0:800), $
    xrange=[0.04, 12.0], xstyle=1, xtitle="frequency [kHz]", $
    yrange=[1.e-3, 10.0], ystyle=1, /ylog, ytitle="relative power"

  wait, 0.2

endwhile

done:
close, 1
end
