/* vgpws_wf_list.c
/*
/* Example 'C' code to extract Voyager PWS waveform samples
/* and list them as hexidecimal values along with a time offset.
/*
/* Input is read from stdin and results are sent to stdout.
/* Typical usage would be:
/*
/*   pwswfrmlist < C1234512.DAT
*/

#include <stdio.h>

int
main (int argc, char *argv[])
{

  unsigned char rec[1024];              /* 1024 bytes per record */
  int line;                             /* SCLK line count (60 ms) */
  double offset;                        /* time offset in seconds */
  int sample;                           /* an extracted wfrm sample */
  int i;                                /* loop index */
  char hex[] =                          /* hex characters */
    {'0','1','2','3','4','5','6','7','8','9','A','B','C','D','E','F'};

  /* Read the first (Engineering header) record. */
  if (fread (rec, sizeof(rec), 1, stdin) != 1) {
    fprintf (stderr, "Error reading engineering header record\n");
    return -1;
  }

  /* The most useful (and reliable) information from the file header is
  /* the PWS ASCII label field which was added to the original EDR
  /* header beginning at byte 249.  The format of this field is as
  /* follows:
  /*
  /*   VOYAGER-n PWS  n/nnnnn.nn yyyy-mm-ddThh:mm:ss.sssZ\0\0
  /*
  /* where VOYAGER-n is either VOYAGER-1 or VOYAGER-2;
  /* n/nnnnn.nn is the spacecraft clock value (partition/mod16.mod60);
  /* yyyy-mm-ddThh:mm:ss.sssZ is the spacecraft event time (SCET) of the
  /*   beginning of the 48-second interval (mod60 count).
  /*
  /* Print the information from the PWS ASCII label field.  This should
  /* be parsed and saved for later use in less-trivial applications.
  /* Note that byte 249 is at 'C' index 248. */
  fprintf (stdout, "%s\r\n", rec + 248);

  /* Now loop through all remaining data records. */
  while (fread (rec, sizeof(rec), 1, stdin)) {

    /* Some files may have missing records, so the value of the SCLK
    /* line count should be used to determine the offset time of any
    /* record.  The 2-byte MSB (big-endian) integer value for the line
    /* count occupies bytes 23 and 24 of every data record.  The
    /* following method of extracting the 2-byte value should be
    /* independent of the host byte ordering. */
    line = ( (int)rec[22] << 8 ) | (int)rec[23];

    /* Time offset in seconds, relative to the value in the header,
    /* of the first sample in this record: */
    offset = 0.06 * ( line - 1 );

    fprintf (stdout, " %03i %5.2f ", line, offset);

    /* Now loop through all 800 bytes (1600 4-bit samples) beginning at
    /* byte 221 in the record */
    for (i = 220; i < 220 + 800; i++) {
      sample = (int)rec[i] >> 4;
      fprintf (stdout, "%c", hex[sample]);
      sample = (int)rec[i] & 0x0f;
      fprintf (stdout, "%c", hex[sample]);
    }
    fprintf (stdout, "\r\n");

  } /* while data successfully read from standard input */

  /* In some applications it may be useful to skip zero-filled or
  /* duplicate records, or skip the first 16 samples of each line.
  /* If the next step is an FFT, it may be useful to map the values
  /* to a set of floating point numbers ranging from, say, -7.5 to
  /* +7.5.  Essentially, however, this example extracts all data for
  /* most purposes.
  */

  return 0;

} /* pwswfrmlist - example program */
